package top.lshaci.framework.utils;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import top.lshaci.framework.utils.enums.FileType;
import top.lshaci.framework.utils.enums.UtilErrorInfo;
import top.lshaci.framework.utils.exception.UtilException;

import java.io.*;
import java.util.List;

/**
 * <p>文件上传工具</p><br>
 *
 * <b>1.0.7: </b>使用hutool替换commons lang3<br>
 *
 * @author lshaci
 * @version 1.0.7
 * @since 1.0.4
 */
@Slf4j
public abstract class FileUploadUtils {

    /**
     * 通过文件后缀名验证上传文件是否允许
     *
     * @param filename      上传文件的文件名
     * @param allowSuffixes 允许上传的后缀名集合(使用大写)
     */
    public static void verifySuffix(String filename, List<String> allowSuffixes) {
        if (StrUtil.isBlank(filename)) {
            throw new UtilException(UtilErrorInfo.file_name_is_blank);
        }
        if (CollectionUtil.isEmpty(allowSuffixes)) {
            throw new UtilException(UtilErrorInfo.file_allow_suffix_is_empty);
        }
        String suffix = filename.substring(filename.lastIndexOf(".") + 1);
        if (StrUtil.isBlank(suffix)) {
            throw new UtilException(UtilErrorInfo.file_suffix_is_blank);
        }
        if (!(allowSuffixes.contains(suffix.trim().toUpperCase()) || allowSuffixes.contains(suffix.trim().toLowerCase()))) {
            throw new UtilException(UtilErrorInfo.file_suffix_not_allow, suffix);
        }
    }

    /**
     * 通过文件头信息验证上传文件是否允许
     *
     * @param inputStream    文件输入流
     * @param allowFileTypes 允许上传的文件类型集合
     */
    public static void verifySuffix(InputStream inputStream, List<FileType> allowFileTypes) {
        if (inputStream == null) {
            throw new UtilException(UtilErrorInfo.file_is_null);
        }
        if (CollectionUtil.isEmpty(allowFileTypes)) {
            throw new UtilException(UtilErrorInfo.file_allow_suffix_is_empty);
        }

        FileType fileType;
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        IoUtil.copy(inputStream, os);
        try (
                ByteArrayInputStream is = new ByteArrayInputStream(os.toByteArray())
        ) {

            fileType = FileTypeUtil.getType(is);
        } catch (Exception e) {
            log.error(UtilErrorInfo.file_upload_failure.getMsg(), e);
            throw new UtilException(UtilErrorInfo.file_upload_failure);
        }
        if (fileType == null || !allowFileTypes.contains(fileType)) {
            throw new UtilException(UtilErrorInfo.file_suffix_not_allow, fileType);
        }
    }

    /**
     * 保存文件到服务器
     *
     * @param dir          保存文件的路径
     * @param saveFileName 保存文件名
     * @param inputStream  文件输入流
     */
    public static void saveFile(String dir, String saveFileName, InputStream inputStream) {
        if (inputStream == null) {
            throw new UtilException(UtilErrorInfo.file_is_null);
        }
        if (StrUtil.isBlank(dir)) {
            throw new UtilException(UtilErrorInfo.file_save_path_is_blank);
        }
        if (StrUtil.isBlank(saveFileName)) {
            throw new UtilException(UtilErrorInfo.file_save_name_is_blank);
        }

        File saveFile = new File(dir, saveFileName);
        if (!saveFile.getParentFile().exists()) {
            saveFile.getParentFile().mkdirs();
        }

        try (
                FileOutputStream os = new FileOutputStream(saveFile);
        ) {
            IoUtil.copy(inputStream, os);
        } catch (IllegalStateException | IOException e) {
            log.error(UtilErrorInfo.file_save_failure.getMsg(), e);
            throw new UtilException(UtilErrorInfo.file_save_failure);
        }
    }

}
