package com.kyant.capsule

import androidx.annotation.FloatRange
import androidx.annotation.IntRange
import androidx.compose.foundation.shape.CornerBasedShape
import androidx.compose.foundation.shape.CornerSize
import androidx.compose.foundation.shape.ZeroCornerSize
import androidx.compose.runtime.Immutable
import androidx.compose.runtime.Stable
import androidx.compose.ui.geometry.*
import androidx.compose.ui.graphics.Outline
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.LayoutDirection.Ltr
import androidx.compose.ui.unit.dp
import androidx.compose.ui.util.fastCoerceIn
import kotlin.math.min

@Immutable
open class ContinuousRoundedRectangle(
    topStart: CornerSize,
    topEnd: CornerSize,
    bottomEnd: CornerSize,
    bottomStart: CornerSize,
    open val continuity: Continuity = Continuity.Default,
) : CornerBasedShape(
    topStart = topStart,
    topEnd = topEnd,
    bottomEnd = bottomEnd,
    bottomStart = bottomStart,
) {
    override fun createOutline(
        size: Size,
        topStart: Float,
        topEnd: Float,
        bottomEnd: Float,
        bottomStart: Float,
        layoutDirection: LayoutDirection,
    ): Outline {
        // rectangle
        if (topStart + topEnd + bottomEnd + bottomStart == 0f) {
            return Outline.Rectangle(size.toRect())
        }

        val (width, height) = size
        val centerX = width * 0.5f
        val centerY = height * 0.5f

        val maxR = min(centerX, centerY)
        val topLeft = (if (layoutDirection == Ltr) topStart else topEnd).fastCoerceIn(0f, maxR)
        val topRight = (if (layoutDirection == Ltr) topEnd else topStart).fastCoerceIn(0f, maxR)
        val bottomRight = (if (layoutDirection == Ltr) bottomEnd else bottomStart).fastCoerceIn(0f, maxR)
        val bottomLeft = (if (layoutDirection == Ltr) bottomStart else bottomEnd).fastCoerceIn(0f, maxR)

        // normal rounded rectangle or circle
        if (
            !continuity.hasSmoothness ||
            (width == height && topLeft == centerX && topLeft == topRight && bottomLeft == bottomRight)
        ) {
            return Outline.Rounded(
                RoundRect(
                    rect = size.toRect(),
                    topLeft = CornerRadius(topLeft),
                    topRight = CornerRadius(topRight),
                    bottomRight = CornerRadius(bottomRight),
                    bottomLeft = CornerRadius(bottomLeft),
                ),
            )
        }

        // continuous rounded rectangle
        val path = continuity.createRoundedRectanglePathSegments(
            width = size.width.toDouble(),
            height = size.height.toDouble(),
            topLeft = topLeft.toDouble(),
            topRight = topRight.toDouble(),
            bottomRight = bottomRight.toDouble(),
            bottomLeft = bottomLeft.toDouble(),
        ).toPath()
        return Outline.Generic(path)
    }

    override fun copy(
        topStart: CornerSize,
        topEnd: CornerSize,
        bottomEnd: CornerSize,
        bottomStart: CornerSize,
    ): ContinuousRoundedRectangle {
        return ContinuousRoundedRectangle(
            topStart = topStart,
            topEnd = topEnd,
            bottomEnd = bottomEnd,
            bottomStart = bottomStart,
            continuity = continuity,
        )
    }

    fun copy(
        topStart: CornerSize = this.topStart,
        topEnd: CornerSize = this.topEnd,
        bottomEnd: CornerSize = this.bottomEnd,
        bottomStart: CornerSize = this.bottomStart,
        continuity: Continuity = this.continuity,
    ): ContinuousRoundedRectangle {
        return ContinuousRoundedRectangle(
            topStart = topStart,
            topEnd = topEnd,
            bottomEnd = bottomEnd,
            bottomStart = bottomStart,
            continuity = continuity,
        )
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is ContinuousRoundedRectangle) return false

        if (topStart != other.topStart) return false
        if (topEnd != other.topEnd) return false
        if (bottomEnd != other.bottomEnd) return false
        if (bottomStart != other.bottomStart) return false
        if (continuity != other.continuity) return false

        return true
    }

    override fun hashCode(): Int {
        var result = topStart.hashCode()
        result = 31 * result + topEnd.hashCode()
        result = 31 * result + bottomEnd.hashCode()
        result = 31 * result + bottomStart.hashCode()
        result = 31 * result + continuity.hashCode()
        return result
    }

    override fun toString(): String {
        return "ContinuousRoundedRectangle(topStart=$topStart, topEnd=$topEnd, bottomEnd=$bottomEnd, " +
                "bottomStart=$bottomStart, cornerSmoothing=$continuity)"
    }
}

@Immutable
data object ContinuousRectangle : ContinuousRoundedRectangle(
    topStart = ZeroCornerSize,
    topEnd = ZeroCornerSize,
    bottomEnd = ZeroCornerSize,
    bottomStart = ZeroCornerSize,
    continuity = G1Continuity,
) {

    override fun toString(): String {
        return "ContinuousRectangle"
    }
}

private val FullCornerSize = CornerSize(50)

@Stable
val ContinuousCapsule: ContinuousRoundedRectangle = ContinuousCapsule()

@Suppress("FunctionName")
@Stable
fun ContinuousCapsule(
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle = ContinuousCapsuleImpl(continuity)

@Immutable
private data class ContinuousCapsuleImpl(
    override val continuity: Continuity = Continuity.Default,
) : ContinuousRoundedRectangle(
    topStart = FullCornerSize,
    topEnd = FullCornerSize,
    bottomEnd = FullCornerSize,
    bottomStart = FullCornerSize,
    continuity = continuity,
) {

    override fun createOutline(
        size: Size,
        topStart: Float,
        topEnd: Float,
        bottomEnd: Float,
        bottomStart: Float,
        layoutDirection: LayoutDirection,
    ): Outline {
        val (width, height) = size
        val (centerX, centerY) = size.center
        val radius = min(centerX, centerY)

        // normal capsule or circle
        if (
            !continuity.hasSmoothness ||
            (width == height && topStart == centerX && topStart == topEnd && bottomStart == bottomEnd)
        ) {
            return Outline.Rounded(
                RoundRect(
                    rect = size.toRect(),
                    radiusX = radius,
                    radiusY = radius,
                ),
            )
        }

        // continuous capsule
        val isHorizontalCapsule = width > height
        return if (isHorizontalCapsule) {
            continuity.createHorizontalCapsuleOutline(size)
        } else {
            continuity.createVerticalCapsuleOutline(size)
        }
    }

    override fun toString(): String {
        return "ContinuousCapsule(cornerSmoothing=$continuity)"
    }
}

@Stable
fun ContinuousRoundedRectangle(
    corner: CornerSize,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        topStart = corner,
        topEnd = corner,
        bottomEnd = corner,
        bottomStart = corner,
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    size: Dp,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        corner = CornerSize(size),
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    @FloatRange(from = 0.0) size: Float,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        corner = CornerSize(size),
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    @IntRange(from = 0, to = 100) percent: Int,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        corner = CornerSize(percent),
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    topStart: Dp = 0.dp,
    topEnd: Dp = 0.dp,
    bottomEnd: Dp = 0.dp,
    bottomStart: Dp = 0.dp,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        topStart = CornerSize(topStart),
        topEnd = CornerSize(topEnd),
        bottomEnd = CornerSize(bottomEnd),
        bottomStart = CornerSize(bottomStart),
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    @FloatRange(from = 0.0) topStart: Float = 0f,
    @FloatRange(from = 0.0) topEnd: Float = 0f,
    @FloatRange(from = 0.0) bottomEnd: Float = 0f,
    @FloatRange(from = 0.0) bottomStart: Float = 0f,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        topStart = CornerSize(topStart),
        topEnd = CornerSize(topEnd),
        bottomEnd = CornerSize(bottomEnd),
        bottomStart = CornerSize(bottomStart),
        continuity = continuity,
    )

@Stable
fun ContinuousRoundedRectangle(
    @IntRange(from = 0, to = 100) topStartPercent: Int = 0,
    @IntRange(from = 0, to = 100) topEndPercent: Int = 0,
    @IntRange(from = 0, to = 100) bottomEndPercent: Int = 0,
    @IntRange(from = 0, to = 100) bottomStartPercent: Int = 0,
    continuity: Continuity = Continuity.Default,
): ContinuousRoundedRectangle =
    ContinuousRoundedRectangle(
        topStart = CornerSize(topStartPercent),
        topEnd = CornerSize(topEndPercent),
        bottomEnd = CornerSize(bottomEndPercent),
        bottomStart = CornerSize(bottomStartPercent),
        continuity = continuity,
    )
