/*
 * Copyright 2019 Christophe Marchand
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package top.marchand.oxygen.maven.project.support.impl;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.KeyStroke;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreePath;
import org.apache.log4j.Logger;
import top.marchand.oxygen.maven.project.support.MavenProjectPlugin;
import top.marchand.oxygen.maven.project.support.impl.nodes.AbstractMavenParentNode;
import top.marchand.oxygen.maven.project.support.impl.nodes.HasIcon;
import top.marchand.oxygen.maven.project.support.impl.nodes.HasValue;
import top.marchand.oxygen.maven.project.support.impl.nodes.ImageHandler;
import top.marchand.oxygen.maven.project.support.impl.nodes.LazyLoadingTreeNode;
import top.marchand.oxygen.maven.project.support.impl.nodes.MavenFileNode;
import top.marchand.oxygen.maven.project.support.impl.nodes.MavenPackageNode;

/**
 *
 * @author cmarchand
 */
public class DlgChooseDependencyResource extends javax.swing.JDialog {
    private static final Logger LOGGER = Logger.getLogger(DlgChooseDependencyResource.class);
    private String result;
    private FilteredTreeModel treeModel;
    private AbstractAction cancelAction = null, okAction = null;


    public DlgChooseDependencyResource(Window owner) {
        super(owner, ModalityType.APPLICATION_MODAL);
        _initComponents();
        loadModel();
    }
    
    private void _initComponents() {
        setTitle("Dependency resources...");
        initComponents();
        tree.setCellRenderer(new MavenTreeCellRenderer());
        tree.addTreeWillExpandListener(new TreeWillExpandListener() {
            @Override
            public void treeWillExpand(TreeExpansionEvent event) throws ExpandVetoException {
                TreePath path = event.getPath();
                if (path.getLastPathComponent() instanceof LazyLoadingTreeNode) {
                  LazyLoadingTreeNode node = (LazyLoadingTreeNode) path.getLastPathComponent();
                  node.expandNode(treeModel, tree);
                }
            }
            @Override
            public void treeWillCollapse(TreeExpansionEvent event) throws ExpandVetoException { }
        });
        setLocationRelativeTo(getParent());
    }
    
    public String getDependencyUrl() {
        result = null;
        setVisible(true);
        return result;
    }
     
    private void loadModel() {
        Map<String,String> map = MavenProjectPlugin.getInstance().getDependenciesMapping();
        RootDependenciesNode root = new RootDependenciesNode("Dependencies");
        map.keySet().forEach((key) -> {
            root.add(new DependencyNode(key, map.get(key)));
        });
        treeModel = new FilteredTreeModel(root) {
            @Override
            public DefaultMutableTreeNode createRootNode() {
                return new RootDependenciesNode("Dependencies");
            }
        };
        tree.setModel(treeModel);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jLabel1 = new javax.swing.JLabel();
        dfFilter = new javax.swing.JTextField();
        pbOk = new javax.swing.JButton();
        pbCancel = new javax.swing.JButton();
        jScrollPane1 = new javax.swing.JScrollPane();
        tree = new javax.swing.JTree();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);

        jLabel1.setText("Filter");

        dfFilter.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                dfFilterKeyTyped(evt);
            }
        });

        pbOk.setText("Ok");
        pbOk.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                pbOkActionPerformed(evt);
            }
        });

        pbCancel.setText("Cancel");
        pbCancel.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                pbCancelActionPerformed(evt);
            }
        });

        tree.setRootVisible(false);
        tree.setShowsRootHandles(true);
        tree.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                treeMouseClicked(evt);
            }
        });
        jScrollPane1.setViewportView(tree);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jScrollPane1)
                    .addGroup(layout.createSequentialGroup()
                        .addGap(0, 319, Short.MAX_VALUE)
                        .addComponent(pbCancel)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(pbOk, javax.swing.GroupLayout.PREFERRED_SIZE, 54, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(jLabel1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(dfFilter)))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(dfFilter, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 354, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(pbOk)
                    .addComponent(pbCancel))
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void okPressed() {
        LOGGER.debug("okPressed()");
        if(tree.getSelectionPath()!=null) {
            Object o = tree.getSelectionPath().getLastPathComponent();
            if(o instanceof MavenFileNode) {
                MavenFileNode mfn = (MavenFileNode)o;
                MavenPackageNode mpn = (MavenPackageNode)mfn.getParent();
                DependencyNode dn = (DependencyNode)mpn.getParent();
                result = "dependency:/"+dn.getValue()+"/"+mpn.getValue().replaceAll("\\.", "/")+"/"+mfn.getValue();
                setVisible(false);
            } else {
                LOGGER.debug("selected node is a "+o.getClass().getName());
            }
        } else {
            LOGGER.warn("getSelectionPath is null");
        }
    }
    @Override
    
    protected JRootPane createRootPane() {
        cancelAction = new AbstractAction("Annuler") {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setVisible(false);
            }
        };
        okAction = new AbstractAction("Ok") {
            @Override
            public void actionPerformed(ActionEvent e) {
                okPressed();
            }
        };
        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0);
        JRootPane rp = new JRootPane();
        rp.registerKeyboardAction(cancelAction, "CANCEL", ks, JComponent.WHEN_IN_FOCUSED_WINDOW);
        rp.registerKeyboardAction(okAction, "OK", KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);
        return rp;
    }
    private void pbOkActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_pbOkActionPerformed
        okPressed();
    }//GEN-LAST:event_pbOkActionPerformed

    private void pbCancelActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_pbCancelActionPerformed
        setVisible(false);
    }//GEN-LAST:event_pbCancelActionPerformed

    private void treeMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_treeMouseClicked
        Object o = tree.getSelectionPath().getLastPathComponent();
        if(evt.getClickCount()==2) {
            if(o instanceof MavenFileNode) {
                okPressed();
            }
        }
    }//GEN-LAST:event_treeMouseClicked

    private void dfFilterKeyTyped(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_dfFilterKeyTyped
        treeModel.filter(dfFilter.getText());
    }//GEN-LAST:event_dfFilterKeyTyped


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTextField dfFilter;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton pbCancel;
    private javax.swing.JButton pbOk;
    private javax.swing.JTree tree;
    // End of variables declaration//GEN-END:variables

    private class RootDependenciesNode extends AbstractMavenParentNode {

        private final String label;
        public RootDependenciesNode(String label) {
            super();
            this.label = label;
        }

        @Override
        public String getValue() {
            return label;
        }

        @Override
        public Icon getIcon() {
            return null;
        }
    }
    
    private class DependencyNode extends LazyLoadingTreeNode implements HasValue, HasIcon {
        private final String label;
        private final String url;
        private final Comparator<MutableTreeNode> comp;
        public DependencyNode(String id, String url) {
            super(url);
            label = id.substring("dependency:/".length());
            this.url=url;
            comp = (MutableTreeNode o1, MutableTreeNode o2) -> {
                if(o1 instanceof AbstractMavenParentNode && o2 instanceof AbstractMavenParentNode) return o1.toString().compareTo(o2.toString());
                else if(o1.getClass().equals(o2.getClass())) return o1.toString().compareTo(o2.toString());
                else if(o1 instanceof AbstractMavenParentNode) return -1;
                else return 1;
            };
        }

        @Override
        public void add(MutableTreeNode newChild) {
            super.add(newChild);
            sortNodes();
        }
        
        public void sortNodes() {
            Collections.sort(children, comp);
        }

        private void loadChildrenFileProtocol(List<MutableTreeNode> ret) {
            try {
                HashMap<String,List<MutableTreeNode>> entries = new HashMap<>();
                File f = new File(new URI(url));
                Path p = f.toPath();
                Files.walkFileTree(p, new FileVisitor<Path>() {
                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                        return FileVisitResult.CONTINUE;
                    }
                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        Path relative = p.relativize(file.getParent());
                        String packageName = relative.toString().replaceAll("[/\\\\]", ".");
                        List<MutableTreeNode> files = entries.get(packageName);
                        if(files==null) {
                            files = new ArrayList<>();
                            entries.put(packageName, files);
                        }
                        files.add(new MavenFileNode(file));
                        return FileVisitResult.CONTINUE;
                    }
                    @Override
                    public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                        return FileVisitResult.CONTINUE;
                    }
                    @Override
                    public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                        return FileVisitResult.CONTINUE;
                    }
                });
                entries.keySet().stream().map((packageName) -> {
                    MavenPackageNode pack = new MavenPackageNode("".equals(packageName) ? "<default>" : packageName);
                    entries.get(packageName).forEach((mfn) -> {
                        pack.add(mfn);
                    });
                    return pack;
                }).forEachOrdered((pack) -> {
                    ret.add(pack);
                });
            } catch(IOException | URISyntaxException ex) {
                LOGGER.debug("while loading "+url, ex);
            }
        }
        private void loadChildrenZipProtocol(List<MutableTreeNode> ret) {
            LOGGER.debug("url is "+url);
            String validUrl = url.substring(4);
            validUrl = validUrl.substring(0, validUrl.length()-2);
            LOGGER.debug("validUrl: "+validUrl);
            try {
                ZipFile zip = new ZipFile(new File(new URI(validUrl)));
                Map<String,MavenPackageNode> packages = new HashMap<>();
                for(Enumeration<? extends ZipEntry> enumer=zip.entries();enumer.hasMoreElements();) {
                    ZipEntry ze = enumer.nextElement();
                    if(ze.isDirectory()) {
                        String packageName = ze.getName().replaceAll("/", ".");
                        MavenPackageNode mp = new MavenPackageNode(packageName);
                        packages.put(packageName, mp);
                    } else {
                        String[] path = ze.getName().split("/");
                        StringJoiner joiner = new StringJoiner(".");
                        for(int i=0;i<path.length-1;i++) {
                            joiner.add(path[i]);
                        }
                        String packageName = joiner.toString();
                        String fileName = path[path.length-1];
                        LOGGER.debug(packageName+"/"+fileName);
                        MavenPackageNode mp = packages.get(packageName);
                        if(mp==null) {
                            mp = new MavenPackageNode(packageName);
                            packages.put(packageName, mp);
                        }
                        mp.addNoSort(new MavenFileNode(Paths.get(fileName)));
                    }
                }
                packages.values().forEach((mpn) -> {
                    if(mpn.getChildCount()>0) {
                        ret.add(mpn);
                        mpn.sortNodes();
                    }
                });
            } catch(IOException | URISyntaxException ex) {
                LOGGER.error("while loading resources from "+url, ex);
            }
        }
        @Override
        public List<MutableTreeNode> loadChildren() {
            List<MutableTreeNode> ret = new ArrayList<>();
            if(url.startsWith("file:")) {
                loadChildrenFileProtocol(ret);
            } else {
                loadChildrenZipProtocol(ret);
            }
            return ret;
        }
        @Override
        public String toString() {
            return getValue();
        }
        @Override
        public String getValue() {
            return label;
        }
        @Override
        public Icon getIcon() {
            return ImageHandler.getInstance().get(ImageHandler.MAVEN_ICON);
        }

        @Override
        public Object clone() {
            DependencyNode ret = (DependencyNode)super.clone();
            for(Enumeration<MutableTreeNode> enumer=children();enumer.hasMoreElements();) {
                DefaultMutableTreeNode child = (DefaultMutableTreeNode)enumer.nextElement();
                ret.add((MutableTreeNode)child.clone());
            }
            return ret;
        }
        
    }

    
    
}
