package com.naivete.framework.admin.boot.service.impl;

import com.naivete.framework.admin.boot.dao.*;
import com.naivete.framework.admin.boot.model.AuthInfoDO;
import com.naivete.framework.admin.boot.model.FunctionDO;
import com.naivete.framework.admin.boot.model.RoleDO;
import com.naivete.framework.admin.boot.model.UserDO;
import com.naivete.framework.admin.boot.service.AuthInfoService;
import com.naivete.framework.common.dao.Result;
import com.naivete.framework.common.dao.ResultList;
import com.naivete.framework.common.util.MD5Utils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 用户认证信息服务实现
 *
 * @author wolf_314
 * @create 2018-05-26 17:20
 **/
@Service("adminAuthInfoService")
public class AuthInfoServiceImpl implements AuthInfoService {

    // 打印日志
    private static Logger logger = LoggerFactory.getLogger(AuthInfoServiceImpl.class);

    // 用户表DAO
    @Resource
    private UserDAO userDAO;
    // 角色表DAO
    @Resource
    private RoleDAO roleDAO;
    // 用户角色映射表DAO
    @Resource
    private UserRoleMappingDAO userRoleMappingDAO;
    // 功能表DAO
    @Resource
    private FunctionDAO functionDAO;
    // 功能角色映射表DAO
    @Resource
    private FunctionRoleMappingDAO functionRoleMappingDAO;

    /**
     * 根据用户编码查询认证信息
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Override
    public Result<AuthInfoDO> queryAuthInfoByUserCode(String userCode) {
        Result<AuthInfoDO> result = new Result<>();
        AuthInfoDO authInfo = new AuthInfoDO();
        if (StringUtils.isBlank(userCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询用户认证信息失败，用户编码为空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            // 查基本信息
            UserDO user = userDAO.selectOneByUserCode(userCode);
            if (null == user) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("查询用户认证信息失败，用户编码无效！");
                logger.error(result.getErrorMessage());
                return result;
            }
            authInfo.setUser(user);
            // 查询角色编码
            List<String> roleCodes = userRoleMappingDAO.selectListByUserCode(userCode);
            if (CollectionUtils.isEmpty(roleCodes)) {
                // 没有任何角色，返回认证信息
                logger.error("没有任何角色1");
                result.setData(authInfo);
                return result;
            }
            // 查询角色
            List<RoleDO> roleList = roleDAO.selectListByRoleCode(roleCodes);
            authInfo.setRoleList(roleList);
            if (CollectionUtils.isEmpty(roleList)) {
                // 没有任何角色，返回认证信息
                logger.error("没有任何角色2");
                result.setData(authInfo);
                return result;
            }
            // 查询功能编码
            List<String> functionCodes = functionRoleMappingDAO.selectListByRoleCode(roleCodes);
            if (CollectionUtils.isEmpty(functionCodes)) {
                // 没有任何功能，返回认证信息
                logger.error("没有任何功能1");
                result.setData(authInfo);
                return result;
            }
            // 查询功能
            List<FunctionDO> functionList = functionDAO.selectListByFunctionCode(functionCodes);
            authInfo.setFunctionList(functionList);
            if (CollectionUtils.isEmpty(roleList)) {
                // 没有任何功能，返回认证信息
                logger.error("没有任何功能2");
                result.setData(authInfo);
                return result;
            }
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询用户认证信息异常，请稍后重试！");
            logger.error(result.getErrorMessage() + e.getMessage());
            return result;
        }
        result.setData(authInfo);
        return result;
    }

    /**
     * 登录验证
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Override
    public Result<UserDO> login(String userCode, String password) {
        Result<UserDO> result = new Result<>();
        if (StringUtils.isEmpty(userCode) || StringUtils.isEmpty(password)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("用户名或密码空!");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            password = MD5Utils.password(password);
            UserDO u = new UserDO();
            u.setUserCode(userCode);
            u.setPassword(password);
            // 查基本信息
            UserDO user = userDAO.selectOneByLogin(u);
            if (null == user) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("用户名或密码空!");
                logger.error(result.getErrorMessage());
                return result;
            }
            result.setData(user);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("登录验证异常，请稍后重试！");
            logger.error(result.getErrorMessage() + e.getMessage());
            return result;
        }
        return result;
    }

    /**
     * 根据用户编码查询角色编码
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/26 17:55
     */
    @Override
    public ResultList<String> queryRoleCodeByUserCode(String userCode) {
        ResultList<String> result = new ResultList<>();
        if (StringUtils.isBlank(userCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询角色编码失败，用户编码空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            // 查询角色编码
            List<String> roleCodes = userRoleMappingDAO.selectListByUserCode(userCode);
            result.setDatalist(roleCodes);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询角色编码异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
        }
        return result;
    }


}
