package com.naivete.framework.admin.boot.service.impl;

import com.naivete.framework.admin.boot.dao.FunctionRoleMappingDAO;
import com.naivete.framework.admin.boot.dao.RoleDAO;
import com.naivete.framework.admin.boot.dao.UserRoleMappingDAO;
import com.naivete.framework.admin.boot.model.FunctionRoleMappingDO;
import com.naivete.framework.admin.boot.model.RoleDO;
import com.naivete.framework.admin.boot.service.RoleService;
import com.naivete.framework.common.dao.Query;
import com.naivete.framework.common.dao.Result;
import com.naivete.framework.common.dao.ResultList;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 角色服务接口
 *
 * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
 * @create 2018/8/10 23:12
 **/
@Service("adminRoleService")
public class RoleServiceImpl implements RoleService {

    // 打印日志
    private static Logger logger = LoggerFactory.getLogger(RoleServiceImpl.class);

    // 角色表DAO
    @Resource
    private RoleDAO roleDAO;
    // 功能角色映射表DAO
    @Resource
    private FunctionRoleMappingDAO functionRoleMappingDAO;
    // 用户角色映射表DAO
    @Resource
    private UserRoleMappingDAO userRoleMappingDAO;
    // 事务
    @Resource
    private TransactionTemplate transactionTemplate;

    /**
     * 分页查询角色列表
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:12
     **/
    @Override
    public ResultList<RoleDO> queryList(Query<RoleDO> query) {
        ResultList<RoleDO> result = new ResultList<>();
        if (null == query || null == query.getData()) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询角色失败，参数空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            result.setTotal(roleDAO.selectListCount(query));
            result.setDatalist(roleDAO.selectList(query));
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询角色异常," + e.getMessage());
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 根据用户编码查询角色
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:12
     **/
    @Override
    public ResultList<RoleDO> queryListByUserCode(String userCode) {
        ResultList<RoleDO> result = new ResultList<>();
        if (StringUtils.isBlank(userCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询角色失败，参数空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            List<String> roleCodes = userRoleMappingDAO.selectListByUserCode(userCode);
            if (!CollectionUtils.isEmpty(roleCodes)) {
                result.setDatalist(roleDAO.selectListByRoleCode(roleCodes));
            }
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询角色异常," + e.getMessage());
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 查询所有角色
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:12
     **/
    @Override
    public ResultList<RoleDO> queryAll() {
        ResultList<RoleDO> result = new ResultList<>();
        try {
            result.setDatalist(roleDAO.selectAll());
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询所有角色异常," + e.getMessage());
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 保存角色权限映射
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:12
     **/
    @Override
    public Result<Void> saveRoleMapping(final String[] funs, final String role) {
        if (StringUtils.isBlank(role)) {
            return new Result<>();
        }
        return transactionTemplate.execute(status -> {
            Result<Void> result = new Result<>();
            try {
                // 根据角色编码删除映射
                functionRoleMappingDAO.deleteByRole(role);
                // 添加角色功能映射
                if (null != funs && funs.length > 0) {
                    List<FunctionRoleMappingDO> list = new ArrayList<>();
                    for (int i = 0; i < funs.length; i++) {
                        FunctionRoleMappingDO mapping = new FunctionRoleMappingDO();
                        mapping.setFunctionCode(funs[i]);
                        mapping.setRoleCode(role);
                        list.add(mapping);
                    }
                    functionRoleMappingDAO.insertBatch(list);
                }
            } catch (Exception e) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("保存角色权限异常，请稍后重试！");
                status.setRollbackOnly();
                logger.error(result.getErrorMessage());
            }
            return result;
        });
    }

    /**
     * 添加单条角色信息
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/25 22:35
     */
    @Override
    public Result<Void> addOne(RoleDO role) {
        Result<Void> result = new Result<>();
        try {
            int num = roleDAO.insert(role);
            if (num != 1) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("添加单条角色信息失败受影响行不等于1！");
                logger.error(result.getErrorMessage());
            }
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("添加单条角色信息异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
        }
        return result;
    }

    /**
     * 查询角色编码最大的一条角色信息
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/25 22:38
     */
    @Override
    public Result<RoleDO> queryMostBigRoleCode() {
        Result<RoleDO> result = new Result<>();
        try {
            RoleDO role = roleDAO.selectMostBigRoleCode();
            result.setData(role);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询角色编码最大的一条角色信息异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
        }
        return result;
    }

    /**
     * 根据角色编码删除角色信息
     * 1、角色表
     * 2、角色功能映射表
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/25 23:27
     */
    @Override
    public Result<Void> removeRoleByRoleCode(final String roleCode) {
        Result<Void> result = new Result<>();
        if (StringUtils.isBlank(roleCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("删除角色失败，角色ID空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return transactionTemplate.execute(status -> {
            Result<Void> result1 = new Result<>();
            try {
                // 删除角色
                roleDAO.deleteByRoleCode(roleCode);
                // 删除角色功能映射
                functionRoleMappingDAO.deleteByRole(roleCode);
                // 删角色用户映射
                userRoleMappingDAO.deleteByRole(roleCode);
            } catch (Exception e) {
                result1.setSuccess(Boolean.FALSE);
                result1.setErrorMessage("删除角色异常：" + e.getMessage());
                logger.error(result1.getErrorMessage());
                status.setRollbackOnly();
            }
            return result1;
        });
    }

    /**
     * 更新角色信息
     * 只允许更新名称和备注
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/26 0:55
     */
    @Override
    public Result<Void> modify(RoleDO role) {
        Result<Void> result = new Result<>();
        try {
            roleDAO.update(role);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("更新角色信息异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
        }
        return result;
    }

}
