package com.naivete.framework.admin.boot.service.impl;

import com.naivete.framework.admin.boot.constants.UserActiveStatusConstants;
import com.naivete.framework.admin.boot.dao.UserDAO;
import com.naivete.framework.admin.boot.dao.UserRoleMappingDAO;
import com.naivete.framework.admin.boot.model.UserDO;
import com.naivete.framework.admin.boot.model.UserRoleMappingDO;
import com.naivete.framework.admin.boot.service.UserService;
import com.naivete.framework.common.dao.Query;
import com.naivete.framework.common.dao.Result;
import com.naivete.framework.common.dao.ResultList;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 用户服务接口
 *
 * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
 * @create 2018/8/10 23:14
 **/
@Service("adminUserService")
public class UserServiceImpl implements UserService {

    // 打印日志
    private static Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);

    // 用户表DAO
    @Resource
    private UserDAO userDAO;
    // 用户角色映射表DAO
    @Resource
    private UserRoleMappingDAO userRoleMappingDAO;
    // 事务
    @Resource
    private TransactionTemplate transactionTemplate;

    /**
     * 分页查询用户列表
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:15
     **/
    @Override
    public ResultList<UserDO> queryList(Query<UserDO> query) {
        ResultList<UserDO> result = new ResultList<>();
        if (null == query || null == query.getData()) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询用户失败，参数空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            result.setTotal(userDAO.selectListCount(query));
            result.setDatalist(userDAO.selectList(query));
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询用户异常," + e.getMessage());
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 启用或禁用用户
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/26 15:54
     */
    @Override
    public Result<Void> forbiddenOrStartUser(UserDO user) {
        Result<Void> result = new Result<>();
        /** 参数校验 **/
        if (null == user) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("参数异常，用户信息空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        if (StringUtils.isBlank(user.getUserCode())) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("参数异常，用户编码空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        if (null == UserActiveStatusConstants.getEnum(UserActiveStatusConstants.class, user.getIsActive())) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("参数异常，启用状态异常：" + user.getIsActive());
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            user.setIsActive(user.getIsActive().equals(UserActiveStatusConstants.Y.name()) ? UserActiveStatusConstants.N.name() : UserActiveStatusConstants.Y.name());
            int num = userDAO.forbiddenOrStartUser(user);
            if (num != 1) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("启用或禁用用户失败，修改记录不等于1！");
                logger.error(result.getErrorMessage());
                return result;
            }
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("启用或禁用用户异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
        }
        return result;
    }

    /**
     * 根据用户编码查询
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @date 2018/8/26 21:31
     */
    @Override
    public Result<UserDO> queryByUserCode(String userCode) {
        Result<UserDO> result = new Result<>();
        if (StringUtils.isEmpty(userCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询失败，用户编码空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            UserDO user = userDAO.selectOneByUserCode(userCode);
            result.setData(user);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("根据用户编码查询异常：" + e.getMessage());
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 保存用户角色映射
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:12
     **/
    @Override
    public Result<Void> saveUserRoleMapping(String[] roles, String userCode) {
        if (StringUtils.isBlank(userCode)) {
            return new Result<>();
        }
        return transactionTemplate.execute(status -> {
            Result<Void> result = new Result<>();
            try {
                // 根据用户编码删除映射
                userRoleMappingDAO.deleteByUserCode(userCode);
                // 新增映射关系
                if (null != roles && roles.length > 0) {
                    List<UserRoleMappingDO> list = new ArrayList<>();
                    for (int i = 0; i < roles.length; i++) {
                        UserRoleMappingDO mapping = new UserRoleMappingDO();
                        mapping.setRoleCode(roles[i]);
                        mapping.setUserCode(userCode);
                        list.add(mapping);
                    }
                    userRoleMappingDAO.insertBatch(list);
                }
            } catch (Exception e) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("保存角色权限异常，请稍后重试！");
                status.setRollbackOnly();
                logger.error(result.getErrorMessage());
            }
            return result;
        });
    }

}
