package com.naivete.framework.admin.boot.config;

import at.pollux.thymeleaf.shiro.dialect.ShiroDialect;
import com.naivete.framework.admin.boot.config.annotation.EnableAdminBootConfiguration;
import org.apache.shiro.cache.CacheManager;
import org.apache.shiro.cache.MemoryConstrainedCacheManager;
import org.apache.shiro.codec.Base64;
import org.apache.shiro.mgt.SecurityManager;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.spring.web.ShiroFilterFactoryBean;
import org.apache.shiro.web.mgt.CookieRememberMeManager;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.apache.shiro.web.servlet.SimpleCookie;
import org.mybatis.spring.annotation.MapperScan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * AdminBootAutoConfiguration
 *
 * @author luoshen
 * @date 2018/9/17 15:43
 */
@Configuration
@ConditionalOnBean(annotation = EnableAdminBootConfiguration.class)
@EnableConfigurationProperties(value = AdminBootProperteis.class)
@MapperScan("com.naivete.framework.admin.boot.dao")
@ComponentScan({"com.naivete.framework.admin.boot.controller", "com.naivete.framework.admin.boot.service"})
public class AdminBootAutoConfiguration {

    // 打印日志
    private static Logger logger = LoggerFactory.getLogger(AdminBootAutoConfiguration.class);

    @Resource
    private AdminBootProperteis adminBootProperteis;

    /**
     * ShiroFilterFactoryBean 处理拦截资源文件问题。
     * 注意：单独一个ShiroFilterFactoryBean配置是或报错的，因为在初始化ShiroFilterFactoryBean的时候需要注入：SecurityManager
     * <p>
     * Filter Chain定义说明
     * 1、一个URL可以配置多个Filter，使用逗号分隔
     * 2、当设置多个过滤器时，全部验证通过，才视为通过
     * 3、部分过滤器可指定参数，如perms，roles
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "shiroFilter")
    public ShiroFilterFactoryBean shiroFilter(SecurityManager securityManager) {
        logger.info("init shiroFilter begin");
        ShiroFilterFactoryBean shiroFilterFactoryBean = new ShiroFilterFactoryBean();
        // 必须设置 SecurityManager
        shiroFilterFactoryBean.setSecurityManager(securityManager);
        // 如果不设置默认会自动寻找Web工程根目录下的"/login.jsp"页面
        shiroFilterFactoryBean.setLoginUrl("/toLogin");
        // 登录成功后要跳转的链接
        shiroFilterFactoryBean.setSuccessUrl("/index");
        // 未授权界面;
        shiroFilterFactoryBean.setUnauthorizedUrl("/unauthorized");
        // 权限控制map.
        Map<String, String> filterChainDefinitionMap = new LinkedHashMap<>();
        filterChainDefinitionMap.put("/login", "anon");
        filterChainDefinitionMap.put("/logout", "anon");
        filterChainDefinitionMap.put("/admin-boot/**", "anon");
        String[] anons = adminBootProperteis.getAnons();
        for (String anon : anons) {
            logger.info("anon={}", anon);
            filterChainDefinitionMap.put(anon, "anon");
        }
        filterChainDefinitionMap.put("/**", "authc");
        shiroFilterFactoryBean.setFilterChainDefinitionMap(filterChainDefinitionMap);
        logger.info("init shiroFilter end");
        return shiroFilterFactoryBean;
    }

    /**
     * cacheManager Bean
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "cacheManager")
    public CacheManager cacheManager() {
        return new MemoryConstrainedCacheManager();
    }

    /**
     * shiroRealm Bean
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "shiroRealm")
    public AuthorizingRealm shiroRealm(CacheManager cacheManager) {
        ShiroRealm shiroRealm = new ShiroRealm();
        shiroRealm.setCacheManager(cacheManager);
        return shiroRealm;
    }

    /**
     * rememberMeCookie Bean
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "rememberMeCookie")
    public SimpleCookie rememberMeCookie() {
        logger.info("maxAge={}", adminBootProperteis.getMaxAge());
        // 这个参数是cookie的名称，对应前端的checkbox的name = rememberMe
        SimpleCookie simpleCookie = new SimpleCookie("rememberMe");
        // 记住我cookie生效时间30天 ,单位秒
        simpleCookie.setMaxAge(adminBootProperteis.getMaxAge());
        return simpleCookie;
    }

    /**
     * rememberMeManager Bean
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "rememberMeManager")
    public CookieRememberMeManager rememberMeManager(SimpleCookie rememberMeCookie) {
        logger.info("cipherKey={}", adminBootProperteis.getCipherKey());
        CookieRememberMeManager cookieRememberMeManager = new CookieRememberMeManager();
        cookieRememberMeManager.setCookie(rememberMeCookie);
        // rememberMe cookie加密的密钥 建议每个项目都不一样 默认AES算法 密钥长度(128 256 512 位)
        cookieRememberMeManager.setCipherKey(Base64.decode(adminBootProperteis.getCipherKey()));
        return cookieRememberMeManager;
    }

    /**
     * SecurityManager
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "securityManager")
    public SecurityManager securityManager(CookieRememberMeManager rememberMeManager, AuthorizingRealm shiroRealm) {
        DefaultWebSecurityManager securityManager = new DefaultWebSecurityManager();
        // 设置realm.
        securityManager.setRealm(shiroRealm);
        // 注入记住我管理器;
        securityManager.setRememberMeManager(rememberMeManager);
        return securityManager;
    }

    /**
     * ShiroDialect，为了在thymeleaf里使用shiro的标签的bean
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:26
     **/
    @Bean(name = "shiroDialect")
    public ShiroDialect shiroDialect() {
        return new ShiroDialect();
    }

}
