package com.naivete.framework.admin.boot.service.impl;

import com.naivete.framework.admin.boot.dao.FunctionDAO;
import com.naivete.framework.admin.boot.dao.FunctionRoleMappingDAO;
import com.naivete.framework.admin.boot.dao.RoleDAO;
import com.naivete.framework.admin.boot.dao.UserRoleMappingDAO;
import com.naivete.framework.admin.boot.model.FunctionDO;
import com.naivete.framework.admin.boot.model.RoleDO;
import com.naivete.framework.admin.boot.service.FunctionService;
import com.naivete.framework.common.dao.Result;
import com.naivete.framework.common.dao.ResultList;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 功能权限服务
 *
 * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
 * @create 2018/8/10 23:18
 **/
@Service("adminFunctionService")
public class FunctionServiceImpl implements FunctionService {

    // 打印日志
    private static Logger logger = LoggerFactory.getLogger(FunctionServiceImpl.class);

    // 角色表DAO
    @Resource
    private RoleDAO roleDAO;
    // 用户角色映射表DAO
    @Resource
    private UserRoleMappingDAO userRoleMappingDAO;
    // 功能表DAO
    @Resource
    private FunctionDAO functionDAO;
    // 功能角色映射表DAO
    @Resource
    private FunctionRoleMappingDAO functionRoleMappingDAO;
    // 事务
    @Resource
    private TransactionTemplate transactionTemplate;

    /**
     * 根据用户编码查询所有权限信息
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:13
     **/
    @Override
    public ResultList<FunctionDO> queryListByUserCode(String userCode) {
        ResultList<FunctionDO> result = new ResultList<>();
        if (StringUtils.isBlank(userCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询权限失败，用户编码为空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            // 查询角色编码
            List<String> roleCodes = userRoleMappingDAO.selectListByUserCode(userCode);
            if (CollectionUtils.isEmpty(roleCodes)) {
                // 没有任何角色，返回认证信息
                return result;
            }
            // 查询角色
            List<RoleDO> roleList = roleDAO.selectListByRoleCode(roleCodes);
            if (CollectionUtils.isEmpty(roleList)) {
                // 没有任何角色，返回认证信息
                return result;
            }
            // 查询功能编码
            List<String> functionCodes = functionRoleMappingDAO.selectListByRoleCode(roleCodes);
            if (CollectionUtils.isEmpty(functionCodes)) {
                // 没有任何功能，返回认证信息
                return result;
            }
            // 查询功能
            List<FunctionDO> functionList = functionDAO.selectListByFunctionCode(functionCodes);
            if (CollectionUtils.isEmpty(roleList)) {
                // 没有任何功能，返回认证信息
                return result;
            }
            result.setDatalist(functionList);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询权限异常，请稍后重试！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 查询所有的功能
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:13
     **/
    @Override
    public ResultList<FunctionDO> queryAll() {
        ResultList<FunctionDO> result = new ResultList<>();
        try {
            // 查询功能
            List<FunctionDO> functionList = functionDAO.selectAll();
            result.setDatalist(functionList);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询所有功能异常，请稍后重试！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 查询所有的子功能
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:13
     **/
    @Override
    public ResultList<FunctionDO> queryChildFun(String parentCode) {
        ResultList<FunctionDO> result = new ResultList<>();
        try {
            // 查询子功能
            List<FunctionDO> functionList = functionDAO.selectAllChild(parentCode);
            result.setDatalist(functionList);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询所有子功能异常，请稍后重试！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 新增
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:14
     **/
    @Override
    public Result<Void> addOne(final FunctionDO function) {
        return transactionTemplate.execute(status -> {
            Result<Void> result = new Result<>();
            try {
                int num = functionDAO.insert(function);
                if (num != 1) {
                    result.setSuccess(Boolean.FALSE);
                    result.setErrorMessage("新增功能失败，受影响行不为1！");
                    status.setRollbackOnly();
                    logger.error(result.getErrorMessage());
                }
            } catch (Exception e) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("新增功能异常，请稍后重试！");
                status.setRollbackOnly();
                logger.error(result.getErrorMessage());
            }
            return result;
        });
    }

    /**
     * 修改
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:14
     **/
    @Override
    public Result<Void> modifyOne(FunctionDO function) {
        Result<Void> result = new Result<>();
        try {
            int num = functionDAO.update(function);
            if (num != 1) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("修改功能失败，受影响行不为1！");
                logger.error(result.getErrorMessage());
            }
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("修改功能异常，请稍后重试！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 根据角色编码查询所有权限信息
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:14
     **/
    @Override
    public ResultList<FunctionDO> queryListByRoleCode(String roleCode) {
        ResultList<FunctionDO> result = new ResultList<>();
        if (StringUtils.isBlank(roleCode)) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询权限失败，角色编码为空！");
            logger.error(result.getErrorMessage());
            return result;
        }
        try {
            List<String> roleCodes = new ArrayList<>();
            roleCodes.add(roleCode);
            // 查询功能编码
            List<String> functionCodes = functionRoleMappingDAO.selectListByRoleCode(roleCodes);
            if (CollectionUtils.isEmpty(functionCodes)) {
                return result;
            }
            // 查询功能
            List<FunctionDO> functionList = functionDAO.selectListByFunctionCode(functionCodes);
            result.setDatalist(functionList);
        } catch (Exception e) {
            result.setSuccess(Boolean.FALSE);
            result.setErrorMessage("查询权限失败，请稍后重试！");
            logger.error(result.getErrorMessage());
            return result;
        }
        return result;
    }

    /**
     * 根据编码删除功能
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/10 23:14
     **/
    @Override
    public Result<Void> removeFun(final String code) {
        return transactionTemplate.execute(status -> {
            Result<Void> result = new Result<>();
            try {
                // 删掉以code开头的功能
                functionDAO.deleteByCode(code);
                functionRoleMappingDAO.deleteByFun(code);
            } catch (Exception e) {
                result.setSuccess(Boolean.FALSE);
                result.setErrorMessage("删除功能异常，请稍后重试！");
                status.setRollbackOnly();
                logger.error(result.getErrorMessage());
            }
            return result;
        });
    }

}
