package com.naivete.framework.common.lang;

import com.naivete.framework.common.constant.MargicNumberConstants;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.*;

/**
 * MEnum
 *
 * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
 * @create 2018/8/1 10:59
 **/
@SuppressWarnings("rawtypes")
public abstract class MEnum<T extends MEnum> implements Comparable, Serializable {

    private static final long serialVersionUID = -3452282956818035040L;

    // EMPTY_MAP
    private static final Map<String, MEnum> EMPTY_MAP = Collections.unmodifiableMap(new HashMap<String, MEnum>(0));

    // 日志
    private static Logger logger = LoggerFactory.getLogger(MEnum.class);

    // cEnumClasses
    private static Map<Class<MEnum>, Entry> cEnumClasses = new WeakHashMap<Class<MEnum>, Entry>();

    // iToString
    protected transient String iToString = null;

    // 英文CODE
    private String name;

    // 中文名称
    private String cname;

    // 值
    private Number value;

    // 描述
    private String desc;

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     **/
    public static MEnum create() {
        return create(null, null, null, null);
    }

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     **/
    public static MEnum create(String name) {
        return create(name, null, null, null);
    }

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static MEnum create(String name, Number value) {
        return create(name, value, null, null);
    }

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static MEnum create(Number value, String cname) {
        return create(null, value, cname, null);
    }

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static MEnum create(String name, Number value, String cname) {
        return create(name, value, cname, null);
    }

    /**
     * create
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static MEnum create(String name, Number value, String cname, String desc) {
        MEnum menum = init(name);
        if (StringUtils.isNotEmpty(name)) {
            menum.name = name;
        }
        menum.cname = cname;
        menum.value = value;
        menum.desc = desc;
        return menum;
    }

    /**
     * init
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    private static MEnum init(String name) {
        Class<MEnum> enumClass;
        try {
            enumClass = (Class<MEnum>) ClassUtils.getClass(Thread.currentThread().getContextClassLoader(), getCallerClassName());
            if (enumClass == null) {
                throw new IllegalArgumentException("EnumClass must not be null");
            }

            Entry entry;
            synchronized (MEnum.class) {
                entry = (Entry) cEnumClasses.get(enumClass);
                if (entry == null) {
                    entry = createEntry(enumClass);
                    Map myMap = new WeakHashMap(); // we avoid the (Map) constructor to achieve JDK 1.2 support
                    myMap.putAll(cEnumClasses);
                    myMap.put(enumClass, entry);
                    cEnumClasses = myMap;
                }
            }

            MEnum enumObject = (MEnum) enumClass.newInstance();

            if (StringUtils.isNotEmpty(name)) {
                if (!entry.map.containsKey(name)) {
                    enumObject.name = name;
                    entry.map.put(name, enumObject);

                } else {
                    throw new IllegalArgumentException("The Enum name must be unique, '" + name + "' has already been added");
                }
            }
            //NAME为空的对象在MAP中没有，在对象实例化后，VALUE取LIST中的值，KEY取当前CLASS的FIELD PUT进去
            entry.list.add(enumObject);

            return enumObject;
        } catch (Exception e) {
            logger.error(e.getMessage());
        }

        return null;
    }

    /**
     * getCallerClassName
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    private static String getCallerClassName() {
        StackTraceElement[] callers = new Throwable().getStackTrace();
        String enumClass = MEnum.class.getName();
        for (StackTraceElement caller : callers) {
            String className = caller.getClassName();
            String methodName = caller.getMethodName();
            if (!enumClass.equals(className) && "<clinit>".equals(methodName)) {
                return className;
            }
        }

        throw new IllegalArgumentException("");
    }

    /**
     * getEntry
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    private static <T> Entry getEntry(Class<T> enumClass) {
        if (enumClass == null) {
            throw new IllegalArgumentException("The Enum Class must not be null");
        }
        if (MEnum.class.isAssignableFrom(enumClass) == false) {
            throw new IllegalArgumentException("The Class must be a subclass of Enum");
        }
        Entry entry = (Entry) cEnumClasses.get(enumClass);

        if (entry == null) {
            try {
                Class.forName(enumClass.getName(), true, enumClass.getClassLoader());
                entry = (Entry) cEnumClasses.get(enumClass);
            } catch (Exception e) {
                // Ignore
            }
        }

        return entry;
    }

    /**
     * createEntry
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    private static Entry createEntry(Class enumClass) {
        Entry entry = new Entry();
        Class cls = enumClass.getSuperclass();
        while (cls != null && cls != MEnum.class) {
            Entry loopEntry = (Entry) cEnumClasses.get(cls);
            if (loopEntry != null) {
                entry.list.addAll(loopEntry.list);
                entry.map.putAll(loopEntry.map);
                break;
            }
            cls = cls.getSuperclass();
        }
        return entry;
    }

    /**
     * getEnum
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static <T> MEnum getEnum(Class<T> enumClass, String name) {
        Entry entry = getEntry(enumClass);
        if (entry == null) {
            return null;
        }
        return entry.map.get(name);
    }

    /**
     * getEnum
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static <T> MEnum getEnum(Class<T> enumClass, Number value) {
        if (enumClass == null) {
            throw new IllegalArgumentException("The Enum Class must not be null");
        }
        List<MEnum> list = MEnum.getEnumList(enumClass);
        for (Iterator<MEnum> it = list.iterator(); it.hasNext(); ) {
            MEnum enumeration = (MEnum) it.next();
            if (enumeration.value() == value.intValue()) {
                return enumeration;
            }
        }
        return null;
    }

    /**
     * getEnumByCnName
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static <T> MEnum getEnumByCnName(Class<T> enumClass, String cnName) {
        if (enumClass == null) {
            throw new IllegalArgumentException("The Enum Class must not be null");
        }
        List<MEnum> list = MEnum.getEnumList(enumClass);
        for (Iterator<MEnum> it = list.iterator(); it.hasNext(); ) {
            MEnum enumeration = (MEnum) it.next();
            if (enumeration.cname.equalsIgnoreCase(cnName)) {
                return enumeration;
            }
        }
        return null;
    }


    /**
     * getEnumMap
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static <T> Map<String, MEnum> getEnumMap(Class<T> enumClass) {
        Entry entry = getEntry(enumClass);
        if (entry == null) {
            return EMPTY_MAP;
        }
        return entry.unmodifiableMap;
    }

    /**
     * getEnumList
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public static <T> List getEnumList(Class<T> enumClass) {
        Entry entry = getEntry(enumClass);
        if (entry == null) {
            return Collections.EMPTY_LIST;
        }
        return entry.unmodifiableList;
    }

    /**
     * iterator
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    protected static Iterator<MEnum> iterator(Class<MEnum> enumClass) {
        return MEnum.getEnumList(enumClass).iterator();
    }

    /**
     * readResolve
     *
     * @return
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    protected Object readResolve() {
        Entry entry = (Entry) cEnumClasses.get(getEnumClass());
        if (entry == null) {
            return null;
        }
        return entry.map.get(name());
    }

    /**
     * getEnumClass
     *
     * @return
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    private Class getEnumClass() {
        Class enumClass = getClass();
        synchronized (enumClass) {
            return enumClass;
        }
    }

    /**
     * equals
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final boolean equals(Object other) {
        if (other == this) {
            return true;
        } else if (other == null) {
            return false;
        } else if (other.getClass() == this.getClass()) {
            return this.name.equals(((MEnum) other).name);
        } else {
            if (other.getClass().getName().equals(this.getClass().getName()) == false) {
                return false;
            }
            return true;
        }
    }

    /**
     * compareTo
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public int compareTo(Object other) {
        if (other.equals(this)) {
            return 0;
        }
        if (other.getClass() != this.getClass()) {
            throw new ClassCastException(
                    "Different enum class '" + ClassUtils.getShortClassName(other.getClass()) + "'");
        }
        return this.name.compareTo(((MEnum) other).name);
    }

    /**
     * isIn
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public boolean isIn(Class<T> enumClass, int value) {
        if (enumClass == null) {
            throw new IllegalArgumentException("The Enum Class must not be null");
        }
        @SuppressWarnings("unchecked")
        List<MEnum> list = MEnum.getEnumList(enumClass);
        for (Iterator<MEnum> it = list.iterator(); it.hasNext(); ) {
            MEnum enumeration = (MEnum) it.next();
            if (enumeration.getValue().intValue() == value) {
                return true;
            }
        }
        return false;
    }

    /**
     * toString
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public String toString() {
        if (iToString == null) {
            String shortName = ClassUtils.getShortClassName(getEnumClass());
            iToString = shortName + "[" + name() + "]";
        }
        return iToString;
    }

    /**
     * hashCode
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final int hashCode() {
        return getClass().hashCode() ^ value.hashCode();
    }

    /**
     * value
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final int value() {
        return this.value.intValue();
    }


    /**
     * byteValue
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final byte byteValue() {
        return this.value.byteValue();
    }

    /**
     * shortValue
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final short shortValue() {
        return this.value.shortValue();
    }

    /**
     * longValue
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final long longValue() {
        return this.value.longValue();
    }

    /**
     * name
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public final String name() {
        if (name == null) {
            Class enumClass = getEnumClass();
            Entry entry = (Entry) cEnumClasses.get(getEnumClass());
            entry.populateNames(enumClass);
        }
        return this.name;
    }

    /**
     * getName
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public String getName() {
        return name;
    }

    /**
     * getValue
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public Number getValue() {
        return value;
    }

    /**
     * getCname
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public String getCname() {
        return cname;
    }

    /**
     * getDesc
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     */
    public String getDesc() {
        return desc;
    }


    /**
     * Entry
     *
     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
     * @create 2018/8/1 11:13
     **/
    private static class Entry {

        /**
         * map
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        final Map<String, MEnum> map = new HashMap<String, MEnum>();

        /**
         * unmodifiableMap
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        final Map<String, MEnum> unmodifiableMap = Collections.unmodifiableMap(map);

        /**
         * list
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        final List<MEnum> list = new ArrayList<MEnum>(MargicNumberConstants.n25);

        /**
         * unmodifiableList
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        final List<MEnum> unmodifiableList = Collections.unmodifiableList(list);

        /**
         * Entry
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        protected Entry() {
            super();
        }

        /**
         * populateNames
         *
         * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
         */
        private final void populateNames(Class enumClass) {
            synchronized (enumClass) {
                Field[] fields = enumClass.getFields();
                for (Field field : fields) {
                    /**
                     * modifier
                     * @author wolf_314 ~^o^~ <a href="http://blog.naivete.top/">blog.naivete.top</a>
                     */
                    int modifier = field.getModifiers();
                    /**
                     * 判断
                     */
                    if (Modifier.isPublic(modifier) && Modifier.isFinal(modifier) && Modifier.isStatic(modifier)) {
                        try {
                            Object value = field.get(null);
                            String fname = field.getName();
                            for (MEnum enumObject : unmodifiableList) {
                                if (value != null && value.equals(enumObject) && (enumObject.name == null) && !unmodifiableMap.containsKey(fname)) {
                                    enumObject.name = fname;
                                    map.put(fname, enumObject);
                                    break;
                                }
                            }

                        } catch (Exception e) {
                            logger.error(e.getMessage());
                        }
                    }
                }
            }
        }
    }


}
