package top.snowed.word;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import sun.misc.BASE64Encoder;

import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * html处理工具
 *
 * @author Mr Zou
 * @date 2021-5-27
 */
public class HtmlUtil {

    /**
     * css正则
     */
    private static Pattern patternCss = Pattern.compile("\\.(.*?)\\{(.*?)}");

    /**
     * 非限定宽度
     */
    public static final String CSS_TEMPLATE_ONE = "background: #fff;padding: 20px 50px 50px 50px;display:table;margin:25px auto;" +
            "border:1px solid #c6c6c6;min-width:650px;min-height:700px;max-width:1200px;position:relative;";

    /**
     * 限定宽度
     */
    public static final String CSS_TEMPLATE_TWO = "background: #fff;padding: 20px 50px 50px 50px;" +
            "display:table;margin:25px auto;border:1px solid #c6c6c6;width:700px;min-height:650px;position:relative;";

    /**
     * 页眉模版
     */
    private String headCssTemplate = "min-height:25pt;";

    /**
     * 页脚模版
     */
    private String footCssTemplate = "color:#7b7b7b;position:absolute;width:calc(100% - 100px);bottom:10px;text-align:center;font-size:1px;";

    /**
     * 标识符
     */
    private String identifier = "des-data";

    /**
     * 标识符内容
     */
    private String identifyTheContent = "wordBinding";

    /**
     * 图片处理为Base64
     */
    private boolean baseImage = false;

    /**
     * 网页参数整改
     */
    private boolean htmlParameters = true;

    /**
     * 网页参数合并
     */
    private boolean parameterCombination = true;

    /**
     * 网页css样式添加
     */
    private boolean pageStyle = true;

    /**
     * 网页css样式添加、是否保留原有样式
     */
    private boolean keepStyle = false;

    /**
     * 合并css样式
     */
    private boolean mergeStyle = true;

    /**
     * 需要页眉
     */
    private boolean headerRequired = true;

    /**
     * 需要页脚
     */
    private boolean footerRequired = true;

    /**
     * 码格式
     */
    private String charsetName = "UTF-8";

    /**
     * 网页css模版选择
     */
    private String cssTemplate = CSS_TEMPLATE_TWO;

    /**
     * 网页格式工厂
     */
    private HtmlFactory htmlFactory;

    /**
     * 智能切割
     */
    private boolean smartCutting = true;

    /**
     * 集合属性
     * 页脚、页眉等
     */
    private Map<String, Element> attribute = new HashMap<String, Element>();

    /**
     * 构建
     *
     * @param path 文件夹路径
     */
    public String build(String path) throws Exception {
        return execute(path);
    }

    /**
     * 执行
     *
     * @param path
     * @return
     * @throws Exception
     */
    private String execute(String path) throws Exception {
        Document parse = Jsoup.parse(new File(path), charsetName);
        Element body = null;
        //网页参数整改
        if (htmlParameters) {
            body = transformHtml(parse);
        } else {
            body = parse.getElementsByTag("body").get(0);
        }
        //合并参数整改
        if (parameterCombination) {
            body = reviseHtmlParam(body);
        }
        //转化图片位base64
        if (baseImage) {
            body = imageFormHtml(body, path);
        }
        //css样式增加
        if (pageStyle) {
            body = addCssFormHtml(body);
        }
        //css页面智能分割
        if (smartCutting) {
            body = pageCssSplit(body);
        }
        //css样式合并
        if (mergeStyle) {
            cssFormHtml(parse, path);
        }
        //外接工厂处理
        if (htmlFactory != null) {
            htmlFactory.operationHtml(parse);
        }
        return parse.toString();
    }

    /**
     * 转化文档整改
     * 剔除多余项目
     *
     * @param document
     */
    private Element transformHtml(Document document) {
        Element head = document.getElementsByTag("head").get(0);
        Element body = document.getElementsByTag("body").get(0);
        //删除head
        Elements link = head.getElementsByTag("link");
        for (Element element : link) {
            element.remove();
        }
        //删除页眉
        Elements header = body.getElementsByClass("Section0");
        if (header != null && header.size() > 0) {
            Element element = header.get(0);
            Elements allElements = element.getAllElements();
            if (allElements.size() >= 1) {
                Element e = allElements.get(1);
                if (smartCutting) {
                    e.attributes().remove("style");
                    e.attributes().add("style", headCssTemplate);
                    //修改页眉属性
                    attribute.put("页眉", e);
                    e.remove();
                } else if (!headerRequired) {
                    e.remove();
                }
            }
        }
        //删除页脚
        if (smartCutting || !footerRequired) {
            Elements foot = body.getElementsByClass("页脚");
            for (Element element : foot) {
                element.remove();
            }
        }
        //自定义页脚
        if (smartCutting && footerRequired) {
            Element element = new Element("p");
            element.attributes().put("style", footCssTemplate);
            element.text("第1页 共7页");
            attribute.put("页脚", element);
        }
        //找查警告内容
        Elements java = body.getElementsContainingOwnText("JAVA");
        for (Element element : java) {
            element.parent().remove();
        }
        return body;
    }

    /**
     * 转网页参数
     * 网页修改
     * 合并属性
     *
     * @param body
     * @return
     */
    private Element reviseHtmlParam(Element body) {
        Elements ps = body.getElementsByTag("p");
        for (Element p : ps) {
            String text = p.text();
            boolean matches = Pattern.matches("(.*)&\\{(.*)\\}&(.*)", text);
            if (matches) {
                Elements allElements = p.getAllElements();
                //是否为单项
                boolean singleChoice = Pattern.matches("^&\\{(.*)\\}&$", text);
                if (singleChoice) {
                    //整体替换为一个Span
                    for (int i = allElements.size() - 1; i >= 1; i--) {
                        Element element = allElements.get(i);
                        if (i == 1) {
                            element.text(text);
                        } else {
                            element.remove();
                        }
                    }
                } else {
                    //筛选替换
                    String newStr = "";
                    boolean isStartReplace = false;
                    for (int i = allElements.size() - 1; i >= 1; i--) {
                        Element element = allElements.get(i);
                        String str = element.text();
                        if (str.contains("&")) {
                            if (!isStartReplace) {
                                isStartReplace = true;
                                newStr = str;
                                element.remove();
                            } else {
                                newStr = str + newStr;
                                element.text(newStr);
                                isStartReplace = false;
                            }
                            continue;
                        }
                        if (isStartReplace) {
                            newStr = str + newStr;
                            element.remove();
                        }
                    }
                }
            }
        }
        return body;
    }

    /**
     * 转网页图片
     *
     * @param path
     * @param body
     * @return
     */
    private Element imageFormHtml(Element body, String path) {
        Elements img = body.getElementsByTag("img");
        String filePath = new File(path).getParent();
        for (Element element : img) {
            String uri = element.attributes().get("src");
            String imgPath = filePath + File.separator + uri;
            String imageBase64 = imageToBase64(imgPath);
            if (!imageBase64.startsWith("data:image/png;base64,")) {
                imageBase64 = "data:image/png;base64," + imageBase64;
            }
            element.attributes().remove("src");
            element.attributes().add("src", imageBase64);
        }
        return body;
    }

    /**
     * 图片转Base64
     *
     * @param path
     * @return
     */
    private String imageToBase64(String path) {
        byte[] data = null;
        //读取图片字节数组
        try {
            InputStream in = new FileInputStream(path);
            data = new byte[in.available()];
            in.read(data);
            in.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        //对字节数组进行Base64编码，得到Base64编码的字符串
        BASE64Encoder encoder = new BASE64Encoder();
        String base64Str = encoder.encode(data);
        return base64Str;
    }

    /**
     * 转css到html
     *
     * @param document
     * @param path
     * @return
     */
    private void cssFormHtml(Document document, String path) throws Exception {
        Elements head = document.getElementsByTag("head");
        if (head != null && head.size() > 0) {
            /**
             * 读css文件
             */
            String replaceStr = path.replace(".html", "_styles.css");
            File fileCss = new File(replaceStr);
            if (!fileCss.isFile()) {
                return;
            }
            FileInputStream fileInputStream = new FileInputStream(fileCss);
            InputStreamReader reader = new InputStreamReader(fileInputStream);
            BufferedReader bufferedReader = new BufferedReader(reader);
            StringBuffer stringBuffer = new StringBuffer();
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                stringBuffer.append(line);
            }
            /**
             * 关键字获取
             */
            Matcher matcher = patternCss.matcher(stringBuffer);
            StringBuffer newBuffer = new StringBuffer();
            while (matcher.find()) {
                String group = matcher.group(0);
                newBuffer.append(group);
            }
            /**
             * 装载
             */
            if (newBuffer.length() > 0) {
                Element element = head.get(0);
                element.append("<style>\n" +
                        newBuffer.toString() +
                        "</style>");
            }
            /**
             * 关闭
             */
            bufferedReader.close();
            reader.close();
            fileInputStream.close();
        }
    }

    /**
     * 页面Css分割
     *
     * @param body
     * @return
     */
    private Element pageCssSplit(Element body) {
        Elements children = body.children();
        List<List<Element>> footerMap = null;
        if (footerRequired) {
            footerMap = new ArrayList<List<Element>>();
        }
        for (Element child : children) {
            String classStr = child.attributes().get("class");
            List<Element> foot = null;
            if (footerRequired) {
                foot = new ArrayList<Element>();
            }
            if (classStr.contains("Section")) {
                //寻找子级
                Elements divChildren = child.children();
                int endIndex = -1;
                for (int i = divChildren.size() - 1; i >= 0; i--) {
                    Element divChild = divChildren.get(i);
                    String styleStr = divChild.attributes().get("style");
                    if (styleStr.contains("page-break-before:always") || i == 0) {
                        //处理包裹
                        if (endIndex == -1) {
                            endIndex = divChildren.size() - 1;
                        }
                        Element div = new Element("div");
                        divChild.after(div);
                        div.attributes().put("style", cssTemplate);
                        div.attributes().put(identifier, identifyTheContent);
                        //填充页眉
                        if (headerRequired) {
                            Element element = attribute.get("页眉");
                            if (element != null) {
                                div.appendChild(element.clone());
                            }
                        }
                        //填充内容
                        for (int m = (i == 0 ? i : (i + 1)); m <= endIndex; m++) {
                            Element element = divChildren.get(m);
                            div.appendChild(element);
                        }
                        //添加页脚
                        if (footerRequired) {
                            Element element = attribute.get("页脚");
                            if (element != null) {
                                Element clone = element.clone();
                                foot.add(clone);
                                div.appendChild(clone);
                            }
                        }
                        //重置位移
                        endIndex = i;
                        if (i != 0) {
                            divChild.remove();
                        }
                    }
                }
            }
            if (footerRequired) {
                footerMap.add(foot);
            }
        }
        //页脚处理
        if (footerRequired) {
            int size = 0;
            int count = 1;
            for (List<Element> foot : footerMap) {
                size += foot.size();
            }
            for (List<Element> foot : footerMap) {
                for (int i = foot.size() - 1; i >= 0; i--) {
                    Element element = foot.get(i);
                    element.text("第" + count + "页 共" + size + "页");
                    count++;
                }
            }
        }
        return body;
    }

    /**
     * 增加css样式到html
     *
     * @param body
     * @return
     */
    private Element addCssFormHtml(Element body) {
        //增加身体样式
        String bodyStyle = "";
        if (keepStyle) {
            bodyStyle = body.attributes().get("style");
            if (!bodyStyle.endsWith(";")) {
                bodyStyle += ";";
            }
        }
        body.attributes().remove("style");
        body.attributes().add("style", bodyStyle + "background:#e6e6e6;");
        //增加页样式
        if (!smartCutting) {
            Elements children = body.children();
            for (Element child : children) {
                String classStr = child.attributes().get("class");
                if (classStr.contains("Section")) {
                    String divStyle = "";
                    if (keepStyle) {
                        divStyle = child.attributes().get("style");
                        if (!divStyle.endsWith(";")) {
                            divStyle += ";";
                        }
                    }
                    child.attributes().remove("style");
                    child.attributes().add("style", divStyle + cssTemplate);
                }
            }
        }
        return body;
    }

    /**
     * 设置图片是否转化
     *
     * @param operate
     * @return
     */
    public HtmlUtil setConversionImage(boolean operate) {
        this.baseImage = operate;
        return this;
    }

    /**
     * 设置网页参数整改
     *
     * @param operate
     * @return
     */
    public HtmlUtil setHtmlParameterRectification(boolean operate) {
        this.htmlParameters = operate;
        return this;
    }

    /**
     * 设置网页参数合并
     *
     * @param operate
     * @return
     */
    public HtmlUtil setHtmlParameterCombination(boolean operate) {
        this.parameterCombination = operate;
        return this;
    }

    /**
     * 追加css样式
     *
     * @param operate
     * @return
     */
    public HtmlUtil setHtmlCssAppend(boolean operate) {
        this.pageStyle = operate;
        return this;
    }

    /**
     * 需要页脚
     *
     * @param operate
     * @return
     */
    public HtmlUtil setNeedFoot(boolean operate) {
        this.footerRequired = operate;
        return this;
    }

    /**
     * 需要页眉
     *
     * @param operate
     * @return
     */
    public HtmlUtil setNeedHead(boolean operate) {
        this.headerRequired = operate;
        return this;
    }

    /**
     * 智能切割
     *
     * @param operate
     * @return
     */
    public HtmlUtil setSmartCutting(boolean operate) {
        this.smartCutting = operate;
        return this;
    }

    /**
     * 设置网页格式模版
     *
     * @param sytle
     * @return
     */
    public HtmlUtil setHtmlCssTemplate(String sytle) {
        this.cssTemplate = sytle;
        return this;
    }

    /**
     * 保留html原有样式
     * 追加css时生效
     *
     * @param operate
     * @return
     */
    public HtmlUtil setHtmlCssKeep(boolean operate) {
        this.keepStyle = operate;
        return this;
    }

    /**
     * 设置标识符
     * 智能分割使用
     *
     * @param str
     * @return
     */
    public HtmlUtil setIdentifier(String str) {
        this.identifier = str;
        return this;
    }

    /**
     * 设置标识符内容
     * 智能分割使用
     *
     * @param str
     * @return
     */
    public HtmlUtil setIdentifierContent(String str) {
        this.identifyTheContent = str;
        return this;
    }

    /**
     * 设置、页脚CSS模版
     *
     * @param stencil
     * @return
     */
    public HtmlUtil setFootCssTemplate(String stencil) {
        this.footCssTemplate = stencil;
        return this;
    }

    /**
     * 设置、页脚CSS模版
     *
     * @param stencil
     * @return
     */
    public HtmlUtil setHeadCssTemplate(String stencil) {
        this.headCssTemplate = stencil;
        return this;
    }

    /**
     * 合并css样式
     *
     * @param operate
     * @return
     */
    public HtmlUtil setHtmlCssMerge(boolean operate) {
        this.mergeStyle = operate;
        return this;
    }

    /**
     * 设置编码格式
     *
     * @param charsetName
     * @return
     */
    public HtmlUtil setCharsetName(String charsetName) {
        this.charsetName = charsetName;
        return this;
    }

    /**
     * 设置html工厂
     *
     * @param htmlFactory
     * @return
     */
    public HtmlUtil setHtmlFactory(HtmlFactory htmlFactory) {
        this.htmlFactory = htmlFactory;
        return this;
    }

    /**
     * html工厂接口
     */
    public static abstract class HtmlFactory {

        /**
         * html操作
         *
         * @param html
         */
        abstract void operationHtml(Document html);
    }
}
