package top.snowed.word;

import com.spire.doc.Document;
import com.spire.doc.FileFormat;

import java.io.File;
import java.io.InputStream;

/**
 * 文档工具
 *
 * @author Mr Zou
 * @date 2021-5-26
 */
public class WordUtil {

    /**
     * html转化工厂接口
     */
    private HtmlConversionFactory htmlConversionFactory;

    /**
     * 执行完成后删除文件
     */
    private boolean deleteFiles = true;

    /**
     * 构建
     *
     * @param inputStream 文件流
     */
    public String build(InputStream inputStream) throws Exception {
        return docToHtml(inputStream);
    }

    /**
     * 获取临时文件名称
     */
    private File getTemporaryFile() {
        String fileName = System.nanoTime() + "";
        synchronized (fileName.intern()) {
            String path = System.getProperty("user.dir") + File.separator + ".wordFile" + File.separator + fileName;
            File file = new File(path);
            if (file.isDirectory()) {
                file = getTemporaryFile();
            }
            file.mkdirs();
            return file;
        }
    }

    /**
     * 文档转Html
     *
     * @param inputStream 欲读取文件流
     * @throws Exception
     */
    private String docToHtml(InputStream inputStream) throws Exception {
        /**
         * 文件保存
         */
        Document doc = new Document(inputStream);
        String path = getTemporaryFile().getPath() + File.separator + "word.html";
        doc.saveToFile(path, FileFormat.Html);
        inputStream.close();
        doc.close();
        /**
         * Html转化
         */
        HtmlUtil htmlUtil = new HtmlUtil();
        if (htmlConversionFactory != null) {
            htmlConversionFactory.setHtmlUtil(htmlUtil);
        }
        String build = htmlUtil.build(path);
        /**
         * 文件删除
         */
        if (deleteFiles) {
            deleteDirectory(new File(path).getParent());
        }
        return build;
    }

    /**
     * 删除单个文件
     *
     * @param fileName
     * @return
     */
    private boolean deleteFile(String fileName) {
        File file = new File(fileName);
        if (file.isFile() && file.exists()) {
            file.delete();
            return true;
        } else {
            System.out.println("删除单个文件" + fileName + "失败！");
            return false;
        }
    }

    /**
     * 删除目录
     *
     * @param dir
     * @return
     */
    private boolean deleteDirectory(String dir) {
        // 如果dir不以文件分隔符结尾，自动添加文件分隔符
        if (!dir.endsWith(File.separator)) {
            dir = dir + File.separator;
        }
        File dirFile = new File(dir);
        // 如果dir对应的文件不存在，或者不是一个目录，则退出
        if (!dirFile.exists() || !dirFile.isDirectory()) {
            System.out.println("删除目录失败" + dir + "目录不存在！");
            return false;
        }
        boolean flag = true;
        // 删除文件夹下的所有文件(包括子目录)
        File[] files = dirFile.listFiles();
        for (int i = 0; i < files.length; i++) {
            // 删除子文件
            if (files[i].isFile()) {
                flag = deleteFile(files[i].getAbsolutePath());
                if (!flag) {
                    break;
                }
            }
            // 删除子目录
            else {
                flag = deleteDirectory(files[i].getAbsolutePath());
                if (!flag) {
                    break;
                }
            }
        }
        if (!flag) {
            System.out.println("删除目录失败");
            return false;
        }
        // 删除当前目录
        if (dirFile.delete()) {
            return true;
        } else {
            System.out.println("删除目录" + dir + "失败！");
            return false;
        }
    }

    /**
     * 设置Html工厂
     *
     * @param htmlConversionFactory
     * @return
     */
    public WordUtil setHtmlConversionFactory(HtmlConversionFactory htmlConversionFactory) {
        this.htmlConversionFactory = htmlConversionFactory;
        return this;
    }

    /**
     * Html接口工厂
     */
    public abstract static class HtmlConversionFactory {
        /**
         * 设置Html工具
         *
         * @param htmlUtil
         * @return
         */
        public abstract HtmlUtil setHtmlUtil(HtmlUtil htmlUtil);
    }

    /**
     * 设置文件删除
     *
     * @param operate
     * @return
     */
    public WordUtil setFileDelete(boolean operate) {
        this.deleteFiles = operate;
        return this;
    }

}
