/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import org.apache.commons.math3.complex.Complex;
import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.exception.MathArithmeticException;
import org.apache.commons.math3.exception.MathUnsupportedOperationException;
import org.apache.commons.math3.exception.MaxCountExceededException;
import org.apache.commons.math3.exception.util.Localizable;
import org.apache.commons.math3.exception.util.LocalizedFormats;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SchurTransformer;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.linear.TriDiagonalTransformer;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.Precision;

public class EigenDecomposition {
    private static final double EPSILON = 1.0E-12;
    private byte maxIter = (byte)30;
    private double[] main;
    private double[] secondary;
    private TriDiagonalTransformer transformer;
    private double[] realEigenvalues;
    private double[] imagEigenvalues;
    private ArrayRealVector[] eigenvectors;
    private RealMatrix cachedV;
    private RealMatrix cachedD;
    private RealMatrix cachedVt;
    private final boolean isSymmetric;

    public EigenDecomposition(RealMatrix matrix) throws MathArithmeticException {
        double symTol = (double)(10 * matrix.getRowDimension() * matrix.getColumnDimension()) * Precision.EPSILON;
        this.isSymmetric = MatrixUtils.isSymmetric(matrix, symTol);
        if (this.isSymmetric) {
            this.transformToTridiagonal(matrix);
            this.findEigenVectors(this.transformer.getQ().getData());
        } else {
            SchurTransformer t2 = this.transformToSchur(matrix);
            this.findEigenVectorsFromSchur(t2);
        }
    }

    @Deprecated
    public EigenDecomposition(RealMatrix matrix, double splitTolerance) throws MathArithmeticException {
        this(matrix);
    }

    public EigenDecomposition(double[] main, double[] secondary) {
        this.isSymmetric = true;
        this.main = (double[])main.clone();
        this.secondary = (double[])secondary.clone();
        this.transformer = null;
        int size = main.length;
        double[][] z2 = new double[size][size];
        for (int i2 = 0; i2 < size; ++i2) {
            z2[i2][i2] = 1.0;
        }
        this.findEigenVectors(z2);
    }

    @Deprecated
    public EigenDecomposition(double[] main, double[] secondary, double splitTolerance) {
        this(main, secondary);
    }

    public RealMatrix getV() {
        if (this.cachedV == null) {
            int m3 = this.eigenvectors.length;
            this.cachedV = MatrixUtils.createRealMatrix(m3, m3);
            for (int k2 = 0; k2 < m3; ++k2) {
                this.cachedV.setColumnVector(k2, this.eigenvectors[k2]);
            }
        }
        return this.cachedV;
    }

    public RealMatrix getD() {
        if (this.cachedD == null) {
            this.cachedD = MatrixUtils.createRealDiagonalMatrix(this.realEigenvalues);
            for (int i2 = 0; i2 < this.imagEigenvalues.length; ++i2) {
                if (Precision.compareTo(this.imagEigenvalues[i2], 0.0, 1.0E-12) > 0) {
                    this.cachedD.setEntry(i2, i2 + 1, this.imagEigenvalues[i2]);
                    continue;
                }
                if (Precision.compareTo(this.imagEigenvalues[i2], 0.0, 1.0E-12) >= 0) continue;
                this.cachedD.setEntry(i2, i2 - 1, this.imagEigenvalues[i2]);
            }
        }
        return this.cachedD;
    }

    public RealMatrix getVT() {
        if (this.cachedVt == null) {
            int m3 = this.eigenvectors.length;
            this.cachedVt = MatrixUtils.createRealMatrix(m3, m3);
            for (int k2 = 0; k2 < m3; ++k2) {
                this.cachedVt.setRowVector(k2, this.eigenvectors[k2]);
            }
        }
        return this.cachedVt;
    }

    public boolean hasComplexEigenvalues() {
        for (int i2 = 0; i2 < this.imagEigenvalues.length; ++i2) {
            if (Precision.equals(this.imagEigenvalues[i2], 0.0, 1.0E-12)) continue;
            return true;
        }
        return false;
    }

    public double[] getRealEigenvalues() {
        return (double[])this.realEigenvalues.clone();
    }

    public double getRealEigenvalue(int i2) {
        return this.realEigenvalues[i2];
    }

    public double[] getImagEigenvalues() {
        return (double[])this.imagEigenvalues.clone();
    }

    public double getImagEigenvalue(int i2) {
        return this.imagEigenvalues[i2];
    }

    public RealVector getEigenvector(int i2) {
        return this.eigenvectors[i2].copy();
    }

    public double getDeterminant() {
        double determinant = 1.0;
        for (double lambda : this.realEigenvalues) {
            determinant *= lambda;
        }
        return determinant;
    }

    public RealMatrix getSquareRoot() {
        if (!this.isSymmetric) {
            throw new MathUnsupportedOperationException();
        }
        double[] sqrtEigenValues = new double[this.realEigenvalues.length];
        for (int i2 = 0; i2 < this.realEigenvalues.length; ++i2) {
            double eigen = this.realEigenvalues[i2];
            if (eigen <= 0.0) {
                throw new MathUnsupportedOperationException();
            }
            sqrtEigenValues[i2] = FastMath.sqrt(eigen);
        }
        RealMatrix sqrtEigen = MatrixUtils.createRealDiagonalMatrix(sqrtEigenValues);
        RealMatrix v2 = this.getV();
        RealMatrix vT = this.getVT();
        return v2.multiply(sqrtEigen).multiply(vT);
    }

    public DecompositionSolver getSolver() {
        if (this.hasComplexEigenvalues()) {
            throw new MathUnsupportedOperationException();
        }
        return new Solver(this.realEigenvalues, this.imagEigenvalues, this.eigenvectors);
    }

    private void transformToTridiagonal(RealMatrix matrix) {
        this.transformer = new TriDiagonalTransformer(matrix);
        this.main = this.transformer.getMainDiagonalRef();
        this.secondary = this.transformer.getSecondaryDiagonalRef();
    }

    private void findEigenVectors(double[][] householderMatrix) {
        int i2;
        double[][] z2 = (double[][])householderMatrix.clone();
        int n2 = this.main.length;
        this.realEigenvalues = new double[n2];
        this.imagEigenvalues = new double[n2];
        double[] e2 = new double[n2];
        for (int i3 = 0; i3 < n2 - 1; ++i3) {
            this.realEigenvalues[i3] = this.main[i3];
            e2[i3] = this.secondary[i3];
        }
        this.realEigenvalues[n2 - 1] = this.main[n2 - 1];
        e2[n2 - 1] = 0.0;
        double maxAbsoluteValue = 0.0;
        for (i2 = 0; i2 < n2; ++i2) {
            if (FastMath.abs(this.realEigenvalues[i2]) > maxAbsoluteValue) {
                maxAbsoluteValue = FastMath.abs(this.realEigenvalues[i2]);
            }
            if (!(FastMath.abs(e2[i2]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(e2[i2]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i2 = 0; i2 < n2; ++i2) {
                if (FastMath.abs(this.realEigenvalues[i2]) <= Precision.EPSILON * maxAbsoluteValue) {
                    this.realEigenvalues[i2] = 0.0;
                }
                if (!(FastMath.abs(e2[i2]) <= Precision.EPSILON * maxAbsoluteValue)) continue;
                e2[i2] = 0.0;
            }
        }
        for (int j2 = 0; j2 < n2; ++j2) {
            int m3;
            int its = 0;
            do {
                int i4;
                for (m3 = j2; m3 < n2 - 1; ++m3) {
                    double delta = FastMath.abs(this.realEigenvalues[m3]) + FastMath.abs(this.realEigenvalues[m3 + 1]);
                    if (FastMath.abs(e2[m3]) + delta == delta) break;
                }
                if (m3 == j2) continue;
                if (its == this.maxIter) {
                    throw new MaxCountExceededException((Localizable)LocalizedFormats.CONVERGENCE_FAILED, this.maxIter, new Object[0]);
                }
                ++its;
                double q2 = (this.realEigenvalues[j2 + 1] - this.realEigenvalues[j2]) / (2.0 * e2[j2]);
                double t2 = FastMath.sqrt(1.0 + q2 * q2);
                q2 = q2 < 0.0 ? this.realEigenvalues[m3] - this.realEigenvalues[j2] + e2[j2] / (q2 - t2) : this.realEigenvalues[m3] - this.realEigenvalues[j2] + e2[j2] / (q2 + t2);
                double u2 = 0.0;
                double s2 = 1.0;
                double c2 = 1.0;
                for (i4 = m3 - 1; i4 >= j2; --i4) {
                    double p2 = s2 * e2[i4];
                    double h2 = c2 * e2[i4];
                    if (FastMath.abs(p2) >= FastMath.abs(q2)) {
                        c2 = q2 / p2;
                        t2 = FastMath.sqrt(c2 * c2 + 1.0);
                        e2[i4 + 1] = p2 * t2;
                        s2 = 1.0 / t2;
                        c2 *= s2;
                    } else {
                        s2 = p2 / q2;
                        t2 = FastMath.sqrt(s2 * s2 + 1.0);
                        e2[i4 + 1] = q2 * t2;
                        c2 = 1.0 / t2;
                        s2 *= c2;
                    }
                    if (e2[i4 + 1] == 0.0) {
                        int n3 = i4 + 1;
                        this.realEigenvalues[n3] = this.realEigenvalues[n3] - u2;
                        e2[m3] = 0.0;
                        break;
                    }
                    q2 = this.realEigenvalues[i4 + 1] - u2;
                    t2 = (this.realEigenvalues[i4] - q2) * s2 + 2.0 * c2 * h2;
                    u2 = s2 * t2;
                    this.realEigenvalues[i4 + 1] = q2 + u2;
                    q2 = c2 * t2 - h2;
                    for (int ia = 0; ia < n2; ++ia) {
                        p2 = z2[ia][i4 + 1];
                        z2[ia][i4 + 1] = s2 * z2[ia][i4] + c2 * p2;
                        z2[ia][i4] = c2 * z2[ia][i4] - s2 * p2;
                    }
                }
                if (t2 == 0.0 && i4 >= j2) continue;
                int n4 = j2;
                this.realEigenvalues[n4] = this.realEigenvalues[n4] - u2;
                e2[j2] = q2;
                e2[m3] = 0.0;
            } while (m3 != j2);
        }
        for (i2 = 0; i2 < n2; ++i2) {
            int j3;
            int k2 = i2;
            double p3 = this.realEigenvalues[i2];
            for (j3 = i2 + 1; j3 < n2; ++j3) {
                if (!(this.realEigenvalues[j3] > p3)) continue;
                k2 = j3;
                p3 = this.realEigenvalues[j3];
            }
            if (k2 == i2) continue;
            this.realEigenvalues[k2] = this.realEigenvalues[i2];
            this.realEigenvalues[i2] = p3;
            for (j3 = 0; j3 < n2; ++j3) {
                p3 = z2[j3][i2];
                z2[j3][i2] = z2[j3][k2];
                z2[j3][k2] = p3;
            }
        }
        maxAbsoluteValue = 0.0;
        for (i2 = 0; i2 < n2; ++i2) {
            if (!(FastMath.abs(this.realEigenvalues[i2]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(this.realEigenvalues[i2]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i2 = 0; i2 < n2; ++i2) {
                if (!(FastMath.abs(this.realEigenvalues[i2]) < Precision.EPSILON * maxAbsoluteValue)) continue;
                this.realEigenvalues[i2] = 0.0;
            }
        }
        this.eigenvectors = new ArrayRealVector[n2];
        double[] tmp = new double[n2];
        for (int i5 = 0; i5 < n2; ++i5) {
            for (int j4 = 0; j4 < n2; ++j4) {
                tmp[j4] = z2[j4][i5];
            }
            this.eigenvectors[i5] = new ArrayRealVector(tmp);
        }
    }

    private SchurTransformer transformToSchur(RealMatrix matrix) {
        SchurTransformer schurTransform = new SchurTransformer(matrix);
        double[][] matT = schurTransform.getT().getData();
        this.realEigenvalues = new double[matT.length];
        this.imagEigenvalues = new double[matT.length];
        for (int i2 = 0; i2 < this.realEigenvalues.length; ++i2) {
            if (i2 == this.realEigenvalues.length - 1 || Precision.equals(matT[i2 + 1][i2], 0.0, 1.0E-12)) {
                this.realEigenvalues[i2] = matT[i2][i2];
                continue;
            }
            double x2 = matT[i2 + 1][i2 + 1];
            double p2 = 0.5 * (matT[i2][i2] - x2);
            double z2 = FastMath.sqrt(FastMath.abs(p2 * p2 + matT[i2 + 1][i2] * matT[i2][i2 + 1]));
            this.realEigenvalues[i2] = x2 + p2;
            this.imagEigenvalues[i2] = z2;
            this.realEigenvalues[i2 + 1] = x2 + p2;
            this.imagEigenvalues[i2 + 1] = -z2;
            ++i2;
        }
        return schurTransform;
    }

    private Complex cdiv(double xr, double xi, double yr, double yi) {
        return new Complex(xr, xi).divide(new Complex(yr, yi));
    }

    private void findEigenVectorsFromSchur(SchurTransformer schur) throws MathArithmeticException {
        double[][] matrixT = schur.getT().getData();
        double[][] matrixP = schur.getP().getData();
        int n2 = matrixT.length;
        double norm = 0.0;
        for (int i2 = 0; i2 < n2; ++i2) {
            for (int j2 = FastMath.max(i2 - 1, 0); j2 < n2; ++j2) {
                norm += FastMath.abs(matrixT[i2][j2]);
            }
        }
        if (Precision.equals(norm, 0.0, 1.0E-12)) {
            throw new MathArithmeticException(LocalizedFormats.ZERO_NORM, new Object[0]);
        }
        double r2 = 0.0;
        double s2 = 0.0;
        double z2 = 0.0;
        for (int idx = n2 - 1; idx >= 0; --idx) {
            double t2;
            int i3;
            int l2;
            double p2 = this.realEigenvalues[idx];
            double q2 = this.imagEigenvalues[idx];
            if (Precision.equals(q2, 0.0)) {
                l2 = idx;
                matrixT[idx][idx] = 1.0;
                for (i3 = idx - 1; i3 >= 0; --i3) {
                    double w2 = matrixT[i3][i3] - p2;
                    r2 = 0.0;
                    for (int j3 = l2; j3 <= idx; ++j3) {
                        r2 += matrixT[i3][j3] * matrixT[j3][idx];
                    }
                    if (Precision.compareTo(this.imagEigenvalues[i3], 0.0, 1.0E-12) < 0) {
                        z2 = w2;
                        s2 = r2;
                        continue;
                    }
                    l2 = i3;
                    if (Precision.equals(this.imagEigenvalues[i3], 0.0)) {
                        matrixT[i3][idx] = w2 != 0.0 ? -r2 / w2 : -r2 / (Precision.EPSILON * norm);
                    } else {
                        double x2 = matrixT[i3][i3 + 1];
                        double y2 = matrixT[i3 + 1][i3];
                        q2 = (this.realEigenvalues[i3] - p2) * (this.realEigenvalues[i3] - p2) + this.imagEigenvalues[i3] * this.imagEigenvalues[i3];
                        matrixT[i3][idx] = t2 = (x2 * s2 - z2 * r2) / q2;
                        matrixT[i3 + 1][idx] = FastMath.abs(x2) > FastMath.abs(z2) ? (-r2 - w2 * t2) / x2 : (-s2 - y2 * t2) / z2;
                    }
                    double t3 = FastMath.abs(matrixT[i3][idx]);
                    if (!(Precision.EPSILON * t3 * t3 > 1.0)) continue;
                    for (int j4 = i3; j4 <= idx; ++j4) {
                        double[] dArray = matrixT[j4];
                        int n3 = idx;
                        dArray[n3] = dArray[n3] / t3;
                    }
                }
                continue;
            }
            if (!(q2 < 0.0)) continue;
            l2 = idx - 1;
            if (FastMath.abs(matrixT[idx][idx - 1]) > FastMath.abs(matrixT[idx - 1][idx])) {
                matrixT[idx - 1][idx - 1] = q2 / matrixT[idx][idx - 1];
                matrixT[idx - 1][idx] = -(matrixT[idx][idx] - p2) / matrixT[idx][idx - 1];
            } else {
                Complex result = this.cdiv(0.0, -matrixT[idx - 1][idx], matrixT[idx - 1][idx - 1] - p2, q2);
                matrixT[idx - 1][idx - 1] = result.getReal();
                matrixT[idx - 1][idx] = result.getImaginary();
            }
            matrixT[idx][idx - 1] = 0.0;
            matrixT[idx][idx] = 1.0;
            for (i3 = idx - 2; i3 >= 0; --i3) {
                double ra = 0.0;
                double sa = 0.0;
                for (int j5 = l2; j5 <= idx; ++j5) {
                    ra += matrixT[i3][j5] * matrixT[j5][idx - 1];
                    sa += matrixT[i3][j5] * matrixT[j5][idx];
                }
                double w3 = matrixT[i3][i3] - p2;
                if (Precision.compareTo(this.imagEigenvalues[i3], 0.0, 1.0E-12) < 0) {
                    z2 = w3;
                    r2 = ra;
                    s2 = sa;
                    continue;
                }
                l2 = i3;
                if (Precision.equals(this.imagEigenvalues[i3], 0.0)) {
                    Complex c2 = this.cdiv(-ra, -sa, w3, q2);
                    matrixT[i3][idx - 1] = c2.getReal();
                    matrixT[i3][idx] = c2.getImaginary();
                } else {
                    double x3 = matrixT[i3][i3 + 1];
                    double y3 = matrixT[i3 + 1][i3];
                    double vr = (this.realEigenvalues[i3] - p2) * (this.realEigenvalues[i3] - p2) + this.imagEigenvalues[i3] * this.imagEigenvalues[i3] - q2 * q2;
                    double vi = (this.realEigenvalues[i3] - p2) * 2.0 * q2;
                    if (Precision.equals(vr, 0.0) && Precision.equals(vi, 0.0)) {
                        vr = Precision.EPSILON * norm * (FastMath.abs(w3) + FastMath.abs(q2) + FastMath.abs(x3) + FastMath.abs(y3) + FastMath.abs(z2));
                    }
                    Complex c3 = this.cdiv(x3 * r2 - z2 * ra + q2 * sa, x3 * s2 - z2 * sa - q2 * ra, vr, vi);
                    matrixT[i3][idx - 1] = c3.getReal();
                    matrixT[i3][idx] = c3.getImaginary();
                    if (FastMath.abs(x3) > FastMath.abs(z2) + FastMath.abs(q2)) {
                        matrixT[i3 + 1][idx - 1] = (-ra - w3 * matrixT[i3][idx - 1] + q2 * matrixT[i3][idx]) / x3;
                        matrixT[i3 + 1][idx] = (-sa - w3 * matrixT[i3][idx] - q2 * matrixT[i3][idx - 1]) / x3;
                    } else {
                        Complex c2 = this.cdiv(-r2 - y3 * matrixT[i3][idx - 1], -s2 - y3 * matrixT[i3][idx], z2, q2);
                        matrixT[i3 + 1][idx - 1] = c2.getReal();
                        matrixT[i3 + 1][idx] = c2.getImaginary();
                    }
                }
                t2 = FastMath.max(FastMath.abs(matrixT[i3][idx - 1]), FastMath.abs(matrixT[i3][idx]));
                if (!(Precision.EPSILON * t2 * t2 > 1.0)) continue;
                for (int j6 = i3; j6 <= idx; ++j6) {
                    double[] dArray = matrixT[j6];
                    int n4 = idx - 1;
                    dArray[n4] = dArray[n4] / t2;
                    double[] dArray2 = matrixT[j6];
                    int n5 = idx;
                    dArray2[n5] = dArray2[n5] / t2;
                }
            }
        }
        for (int j7 = n2 - 1; j7 >= 0; --j7) {
            for (int i4 = 0; i4 <= n2 - 1; ++i4) {
                z2 = 0.0;
                for (int k2 = 0; k2 <= FastMath.min(j7, n2 - 1); ++k2) {
                    z2 += matrixP[i4][k2] * matrixT[k2][j7];
                }
                matrixP[i4][j7] = z2;
            }
        }
        this.eigenvectors = new ArrayRealVector[n2];
        double[] tmp = new double[n2];
        for (int i5 = 0; i5 < n2; ++i5) {
            for (int j8 = 0; j8 < n2; ++j8) {
                tmp[j8] = matrixP[j8][i5];
            }
            this.eigenvectors[i5] = new ArrayRealVector(tmp);
        }
    }

    private static class Solver
    implements DecompositionSolver {
        private double[] realEigenvalues;
        private double[] imagEigenvalues;
        private final ArrayRealVector[] eigenvectors;

        private Solver(double[] realEigenvalues, double[] imagEigenvalues, ArrayRealVector[] eigenvectors) {
            this.realEigenvalues = realEigenvalues;
            this.imagEigenvalues = imagEigenvalues;
            this.eigenvectors = eigenvectors;
        }

        public RealVector solve(RealVector b2) {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m3 = this.realEigenvalues.length;
            if (b2.getDimension() != m3) {
                throw new DimensionMismatchException(b2.getDimension(), m3);
            }
            double[] bp = new double[m3];
            for (int i2 = 0; i2 < m3; ++i2) {
                ArrayRealVector v2 = this.eigenvectors[i2];
                double[] vData = v2.getDataRef();
                double s2 = v2.dotProduct(b2) / this.realEigenvalues[i2];
                for (int j2 = 0; j2 < m3; ++j2) {
                    int n2 = j2;
                    bp[n2] = bp[n2] + s2 * vData[j2];
                }
            }
            return new ArrayRealVector(bp, false);
        }

        public RealMatrix solve(RealMatrix b2) {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m3 = this.realEigenvalues.length;
            if (b2.getRowDimension() != m3) {
                throw new DimensionMismatchException(b2.getRowDimension(), m3);
            }
            int nColB = b2.getColumnDimension();
            double[][] bp = new double[m3][nColB];
            double[] tmpCol = new double[m3];
            for (int k2 = 0; k2 < nColB; ++k2) {
                int i2;
                for (i2 = 0; i2 < m3; ++i2) {
                    tmpCol[i2] = b2.getEntry(i2, k2);
                    bp[i2][k2] = 0.0;
                }
                for (i2 = 0; i2 < m3; ++i2) {
                    int j2;
                    ArrayRealVector v2 = this.eigenvectors[i2];
                    double[] vData = v2.getDataRef();
                    double s2 = 0.0;
                    for (j2 = 0; j2 < m3; ++j2) {
                        s2 += v2.getEntry(j2) * tmpCol[j2];
                    }
                    s2 /= this.realEigenvalues[i2];
                    for (j2 = 0; j2 < m3; ++j2) {
                        double[] dArray = bp[j2];
                        int n2 = k2;
                        dArray[n2] = dArray[n2] + s2 * vData[j2];
                    }
                }
            }
            return new Array2DRowRealMatrix(bp, false);
        }

        public boolean isNonSingular() {
            int i2;
            double largestEigenvalueNorm = 0.0;
            for (i2 = 0; i2 < this.realEigenvalues.length; ++i2) {
                largestEigenvalueNorm = FastMath.max(largestEigenvalueNorm, this.eigenvalueNorm(i2));
            }
            if (largestEigenvalueNorm == 0.0) {
                return false;
            }
            for (i2 = 0; i2 < this.realEigenvalues.length; ++i2) {
                if (!Precision.equals(this.eigenvalueNorm(i2) / largestEigenvalueNorm, 0.0, 1.0E-12)) continue;
                return false;
            }
            return true;
        }

        private double eigenvalueNorm(int i2) {
            double re = this.realEigenvalues[i2];
            double im = this.imagEigenvalues[i2];
            return FastMath.sqrt(re * re + im * im);
        }

        public RealMatrix getInverse() {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m3 = this.realEigenvalues.length;
            double[][] invData = new double[m3][m3];
            for (int i2 = 0; i2 < m3; ++i2) {
                double[] invI = invData[i2];
                for (int j2 = 0; j2 < m3; ++j2) {
                    double invIJ = 0.0;
                    for (int k2 = 0; k2 < m3; ++k2) {
                        double[] vK = this.eigenvectors[k2].getDataRef();
                        invIJ += vK[i2] * vK[j2] / this.realEigenvalues[k2];
                    }
                    invI[j2] = invIJ;
                }
            }
            return MatrixUtils.createRealMatrix(invData);
        }
    }
}

