package com.ben.utils.intent

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Bundle
import java.lang.Exception

/**
 * Author: bin.yang
 * Maintainer: bin.yang
 * Date: 2019/6/29 0029
 * Copyright: 2019 Inc. All rights reserved.
 * Desc:意图相关
 */
object IntentUtils {
    /**
     * 根据包名启动app
     *
     * @param context     The Context
     * @param packageName 包名
     */
    fun launchAPP(context: Context, packageName: String?) {
        try {
            val intent = context.applicationContext.packageManager.getLaunchIntentForPackage(
                packageName!!
            )
            if (intent != null) {
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                context.startActivity(intent)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 通过action启动app
     *
     * @param context The Context
     * @param action  action
     */
    fun launchAction(context: Context, action: String?) {
        try {
            val intent = Intent(action)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 通过action和包名启动app
     *
     * @param context The Context
     * @param action  action
     */
    fun launchActionWithPkgName(context: Context, action: String?, pkgName: String?) {
        try {
            val intent = Intent(action)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.setPackage(pkgName)
            context.startActivity(intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 通过action启动app，并设置bundle
     *
     * @param context The Context
     * @param action  广播
     */
    fun launchActionWithBundle(context: Context, action: String?, bundle: Bundle?) {
        try {
            val intent = Intent(action)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.putExtras(bundle!!)
            context.startActivity(intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 启动指定activity
     *
     * @param context
     * @param packageName
     * @param activityName
     */
    fun launchActivity(context: Context, packageName: String?, activityName: String?) {
        try {
            val cn = ComponentName(packageName!!, activityName!!)
            val intent = Intent()
            intent.component = cn
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
            context.startActivity(intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * Activity intent有效性检查
     *
     * @param context
     * @param intent
     * @return true：有效 false:无效
     */
    fun checkActivityIntent(context: Context, intent: Intent?): Boolean {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryIntentActivities(
            intent!!,
            PackageManager.MATCH_DEFAULT_ONLY
        )
        return list.size > 0
    }

    /**
     * 服务 intent有效性检查
     *
     * @param context
     * @param intent
     * @return true：有效 false:无效
     */
    fun checkServiceIntent(context: Context, intent: Intent?): Boolean {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryIntentServices(
            intent!!,
            PackageManager.MATCH_DEFAULT_ONLY
        )
        return list.size > 0
    }

    /**
     * 广播 intent有效性检查
     *
     * @param context
     * @param intent
     * @return true：有效 false:无效
     */
    fun checkReceiverIntent(context: Context, intent: Intent?): Boolean {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryBroadcastReceivers(intent!!, 0)
        return list.size > 0
    }

    /**
     * 通过Activity intent获取包名
     *
     * @param context
     * @param intent
     * @return
     */
    fun getPkgNameByActivityIntent(context: Context, intent: Intent?): String? {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryIntentActivities(intent!!, 0)
        return if (list != null && !list.isEmpty()) {
            list[0].activityInfo.packageName
        } else null
    }

    /**
     * 通过服务 intent获取包名
     *
     * @param context
     * @param intent
     * @return
     */
    fun getPkgNameByServiceIntent(context: Context, intent: Intent?): String? {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryIntentServices(intent!!, 0)
        return if (list != null && !list.isEmpty()) {
            list[0].activityInfo.packageName
        } else null
    }

    /**
     * 通过广播intent获取包名
     *
     * @param context
     * @param intent
     * @return
     */
    fun getPkgNameByReceiverIntent(context: Context, intent: Intent?): String? {
        val packageManager = context.applicationContext.packageManager
        val list = packageManager.queryBroadcastReceivers(
            intent!!, 0
        )
        return if (list != null && !list.isEmpty()) {
            list[0].activityInfo.packageName
        } else null
    }

    /**
     * 启动原生卸载apk的界面
     *
     * @param context
     * @param pkgname
     */
    fun launchAppInstallActivity(context: Context, pkgname: String) {
        val intent = Intent(Intent.ACTION_DELETE, Uri.parse("package:$pkgname"))
        context.startActivity(intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK))
    }
}
