package com.ben.utils.image

import android.graphics.Bitmap
import android.graphics.BitmapShader
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.Paint
import android.graphics.PixelFormat
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.Rect
import android.graphics.RectF
import android.graphics.Shader
import android.graphics.drawable.AdaptiveIconDrawable
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.Drawable
import android.graphics.drawable.NinePatchDrawable
import android.os.Build

/**
 * Author: bin.yang
 * Maintainer: bin.yang
 * Date: 2020/3/10
 * Copyright: 2020 Inc. All rights reserved.
 * Desc:
 */
object ImageUtils {

    fun drawableToBitmap(drawable: Drawable?, config: Boolean): Bitmap? {
        if (drawable == null) {
            return null
        }
        if (drawable is BitmapDrawable) { // 转换成Bitmap
            return drawable.bitmap
        } else if (drawable is NinePatchDrawable) { // .9图片转换成Bitmap
            val bitmap = Bitmap.createBitmap(
                drawable.getIntrinsicWidth(),
                drawable.getIntrinsicHeight(),
                if (config) Bitmap.Config.ARGB_8888 else Bitmap.Config.RGB_565
            )
            val canvas = Canvas(bitmap)
            drawable.setBounds(
                0, 0, drawable.getIntrinsicWidth(),
                drawable.getIntrinsicHeight()
            )
            drawable.draw(canvas)
            return bitmap
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) { // Android8.0自适应图标类型
            if (drawable is AdaptiveIconDrawable) {
                val bitmap = Bitmap.createBitmap(
                    drawable.getIntrinsicWidth(),
                    drawable.getIntrinsicHeight(),
                    if (drawable.getOpacity() != PixelFormat.OPAQUE && config) Bitmap.Config.ARGB_8888 else Bitmap.Config.RGB_565
                )
                val canvas = Canvas(bitmap)
                drawable.setBounds(
                    0, 0, drawable.getIntrinsicWidth(),
                    drawable.getIntrinsicHeight()
                )
                drawable.draw(canvas)
                return bitmap
            }
        } else {
            return null
        }
        return null
    }

    fun getRoundedCornerBitmap(bitmap: Bitmap, roundPx: Float): Bitmap? {
        return try {
            val output = Bitmap.createBitmap(bitmap.width, bitmap.height, Bitmap.Config.ARGB_8888)
            val canvas = Canvas(output)
            val paint = Paint()
            val rect = Rect(0, 0, bitmap.width, bitmap.height)
            val rectF = RectF(Rect(0, 0, bitmap.width, bitmap.height))
            paint.isAntiAlias = true
            canvas.drawARGB(0, 0, 0, 0)
            paint.color = Color.BLACK
            canvas.drawRoundRect(rectF, roundPx, roundPx, paint)
            paint.xfermode = PorterDuffXfermode(PorterDuff.Mode.SRC_IN)
            val src = Rect(0, 0, bitmap.width, bitmap.height)
            canvas.drawBitmap(bitmap, src, rect, paint)
            output
        } catch (e: Exception) {
            bitmap
        }
    }

    fun getCornerBitmap(bitmap: Bitmap, roundPx: Float, baseWidth: Int, baseHeight: Int): Bitmap? {
        return try {
            val output = Bitmap.createBitmap(baseWidth, baseHeight, Bitmap.Config.ARGB_8888)
            val canvas = Canvas(output)
            val paint = Paint()
            val rectF = RectF(Rect(0, 0, baseWidth, baseHeight))
            paint.isAntiAlias = true
            val matrix = Matrix()
            matrix.setScale(baseWidth.toFloat() / bitmap.width, baseHeight.toFloat() / bitmap.height)
            paint.shader = BitmapShader(bitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP)
            paint.shader.setLocalMatrix(matrix)
            canvas.drawRoundRect(rectF, roundPx, roundPx, paint)
            output
        } catch (e: Exception) {
            e.printStackTrace()
            bitmap
        }
    }

    fun drawCornerBitmap(canvas: Canvas, locationRectF: RectF, bitmap: Bitmap, roundPx: Float, baseWidth: Int, baseHeight: Int) {
        return try {
//            val output = Bitmap.createBitmap(baseWidth, baseHeight, Bitmap.Config.RGB_565)
//            val canvas = Canvas(output)
            val paint = Paint()
//            val rectF = RectF(Rect(0, 0, baseWidth, baseHeight))
            paint.isAntiAlias = true
            val matrix = Matrix()
            matrix.setScale(baseWidth.toFloat() / bitmap.width, baseHeight.toFloat() / bitmap.height)
            paint.shader = BitmapShader(bitmap, Shader.TileMode.REPEAT, Shader.TileMode.REPEAT)
            paint.shader.setLocalMatrix(matrix)
            canvas.drawRoundRect(locationRectF, roundPx, roundPx, paint)
//            canvas.drawBitmap()
//            output
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }
}
