package com.yuzhyn.azylee.core.datas.ids;

import com.yuzhyn.azylee.core.datas.collections.ListTool;
import com.yuzhyn.azylee.core.datas.numbers.IntTool;
import com.yuzhyn.azylee.core.datas.numbers.RandomTool;

import java.util.Arrays;
import java.util.List;

import static java.util.stream.Collectors.toList;

/**
 * 雪花算法ID
 */
public class SnowFlake {
    Long MAX = Long.MAX_VALUE;

    //因为二进制里第一个 bit 为如果是 1，那么都是负数，但是我们生成的 id 都是正数，所以第一个 bit 统一都是 0。

    //机器ID  2进制5位  32位减掉1位 31个
    private long workerId;
    //机房ID 2进制5位  32位减掉1位 31个
    private long datacenterId;
    //代表一毫秒内生成的多个id的最新序号  12位 4096 -1 = 4095 个
    private long sequence;
    //设置一个时间初始值    2^41 - 1   差不多可以用69年
    //69年是因为：从0（1970-01-01 08:00:00）开始到最大值（2039-09-07 23:47:35）约69年9个月
    //这里设置初始值为(1609430400000L)：2021-01-01 00:00:00，预计能用到2090年09月
    //临近时间生成的id长度为17位
    private long twepoch = 1609430400000L;
    //5位的机器id
    private long workerIdBits = 5L;
    //5位的机房id
    private long datacenterIdBits = 5L;
    //每毫秒内产生的id数 2 的 12次方
    private long sequenceBits = 12L;
    // 这个是二进制运算，就是5 bit最多只能有31个数字，也就是说机器id最多只能是32以内
    private long maxWorkerId = -1L ^ (-1L << workerIdBits);
    // 这个是一个意思，就是5 bit最多只能有31个数字，机房id最多只能是32以内
    private long maxDatacenterId = -1L ^ (-1L << datacenterIdBits);

    private long workerIdShift = sequenceBits;
    private long datacenterIdShift = sequenceBits + workerIdBits;
    private long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;
    private long sequenceMask = -1L ^ (-1L << sequenceBits);
    //记录产生时间毫秒数，判断是否是同1毫秒
    private long lastTimestamp = -1L;

    public long getWorkerId() {
        return workerId;
    }

    public long getDatacenterId() {
        return datacenterId;
    }

    public long getTimestamp() {
        return System.currentTimeMillis();
    }


    public SnowFlake(long datacenterId, long workerId) {
        this(datacenterId, workerId, 0);
    }

    public SnowFlake(long datacenterId, long workerId, long sequence) {

        // 检查机房id和机器id是否超过31 不能小于0
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(
                    String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }

        if (datacenterId > maxDatacenterId || datacenterId < 0) {

            throw new IllegalArgumentException(
                    String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
        }
        this.workerId = workerId;
        this.datacenterId = datacenterId;
        this.sequence = sequence;
    }

    // 这个是核心方法，通过调用nextId()方法，让当前这台机器上的snowflake算法程序生成一个全局唯一的id
    private synchronized long nextId() {
        // 这儿就是获取当前时间戳，单位是毫秒
        long timestamp = timeGen();
        if (timestamp < lastTimestamp) {

            System.err.printf(
                    "clock is moving backwards. Rejecting requests until %d.", lastTimestamp);
            throw new RuntimeException(
                    String.format("Clock moved backwards. Refusing to generate id for %d milliseconds",
                            lastTimestamp - timestamp));
        }

        // 下面是说假设在同一个毫秒内，又发送了一个请求生成一个id
        // 这个时候就得把seqence序号给递增1，最多就是4096
        if (lastTimestamp == timestamp) {

            // 这个意思是说一个毫秒内最多只能有4096个数字，无论你传递多少进来，
            //这个位运算保证始终就是在4096这个范围内，避免你自己传递个sequence超过了4096这个范围
            sequence = (sequence + 1) & sequenceMask;
            //当某一毫秒的时间，产生的id数 超过4095，系统会进入等待，直到下一毫秒，系统继续产生ID
            if (sequence == 0) {
                timestamp = tilNextMillis(lastTimestamp);
            }

        } else {
            sequence = 0;
        }
        // 这儿记录一下最近一次生成id的时间戳，单位是毫秒
        lastTimestamp = timestamp;
        // 这儿就是最核心的二进制位运算操作，生成一个64bit的id
        // 先将当前时间戳左移，放到41 bit那儿；将机房id左移放到5 bit那儿；将机器id左移放到5 bit那儿；将序号放最后12 bit
        // 最后拼接起来成一个64 bit的二进制数字，转换成10进制就是个long型
        return ((timestamp - twepoch) << timestampLeftShift) |
                (datacenterId << datacenterIdShift) |
                (workerId << workerIdShift) | sequence;
    }

    public synchronized long next() {
        return nextId();
    }

    public synchronized String nexts() {
        return String.valueOf(nextId());
    }

    /**
     * 当某一毫秒的时间，产生的id数 超过4095，系统会进入等待，直到下一毫秒，系统继续产生ID
     *
     * @param lastTimestamp 参数
     * @return 返回 返回
     */
    private long tilNextMillis(long lastTimestamp) {

        long timestamp = timeGen();

        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    //获取当前时间戳
    private long timeGen() {
        return System.currentTimeMillis();
    }

    /**
     * 自动计算数据中心ID和机器ID
     *
     * @param currentDataCenterIds 当前已有的数据中心ID
     * @param currentWorkerIds     当前已有的机器ID
     * @param random               是否随机，否则为顺序生成
     * @return 返回 返回数组[0]-数据中心ID，[1]-机器ID（没有可以指派的ID时，返回-1）
     */
    public static int[] createDataCenterIdAndWorkerId(List<Integer> currentDataCenterIds, List<Integer> currentWorkerIds, boolean random) {
        int[] result = new int[]{-1, -1};
        List<Integer> dataCenterIds = IntTool.createIntegerList(0, 31);
        List<Integer> workerIds = IntTool.createIntegerList(0, 31);
        if (ListTool.ok(currentDataCenterIds)) dataCenterIds.removeAll(currentDataCenterIds);
        if (ListTool.ok(currentWorkerIds)) workerIds.removeAll(currentWorkerIds);

        if (ListTool.ok(dataCenterIds)) {
            if (random) {
                result[0] = dataCenterIds.get(RandomTool.get(0, dataCenterIds.size() - 1));
            } else {
                result[0] = dataCenterIds.get(0);
            }
        }
        if (ListTool.ok(currentWorkerIds)) {
            if (random) {
                result[1] = workerIds.get(RandomTool.get(0, workerIds.size() - 1));
            } else {
                result[1] = workerIds.get(0);
            }
        }
        return result;
    }

    public static void main(String[] args) {

        List<Integer> dcs = Arrays.asList(0, 0, 0, 0);
        List<Integer> wks = Arrays.asList(0, 1, 2, 1);
        for (int i = 0; i < 100; i++) {
            int[] rs = SnowFlake.createDataCenterIdAndWorkerId(dcs, wks, true);
            System.out.println("data-center-id: " + rs[0] + ", worker-id: " + rs[1]);
        }

//        SnowFlake snowFlake = new SnowFlake(100, 100, 0);
//        for (int i = 0; i < 10; i++) {
//            System.out.println(snowFlake.nextId());
//        }
    }
}