/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import java.util.List;

import auto.parcel.AutoParcel;

/**
 * Language specific {@link FullMtgObject} content: texts, descriptions, media.
 */
@AutoParcel
@JsonDeserialize(builder = AutoParcel_Content.Builder.class)
public abstract class Content implements Parcelable {

    public static Builder builder() {
        return new AutoParcel_Content.Builder();
    }

    /**
     * Language (ISO 639-1) of this content item.
     */
    @NonNull
    public abstract String language();

    /**
     * Title.
     */
    @NonNull
    public abstract String title();

    /**
     * Shortened description.
     */
    @NonNull
    public abstract String summary();

    /**
     * Full description.
     */
    @NonNull
    @JsonProperty("desc")
    public abstract String description();

    /**
     * News (typical for museum).
     */
    @Nullable
    public abstract String news();

    /**
     * A message to notify/congratulate the end-user of successful end of guide/quest. Typical
     * for museum, collection, tour.
     */
    @Nullable
    public abstract String closingLine();

    /**
     * MTGObject's playback algorithm settings. Specifies the way data inside MTGObject must be
     * played.
     */
    @Nullable
    public abstract Playback playback();

    /**
     * Download links to offline packages for each supported mobile platform and combination of
     * image, audio and video data formats.
     */
    @Nullable
    public abstract Download download();

    /**
     * Array of images data.
     */
    @Nullable
    public abstract List<Media> images();

    /**
     * Array of audio data.
     */
    @Nullable
    public abstract List<Media> audio();

    /**
     * Array of video data.
     */
    @Nullable
    public abstract List<Media> video();

    /**
     * Array of included object's content.
     */
    @Nullable
    public abstract List<CompactMtgObject> children();

    /**
     * Array of included museum collection's content.
     */
    @Nullable
    public abstract List<CompactMtgObject> collections();

    /**
     * Array of linked object's content.
     */
    @Nullable
    public abstract List<CompactMtgObject> references();

    /**
     * Quiz data.
     */
    @Nullable
    public abstract Quiz quiz();

    /**
     * Number of children.
     */
    @Nullable
    public abstract Integer childrenCount();

    /**
     * Total audio duration.
     */
    @Nullable
    public abstract Integer audioDuration();

    @AutoParcel.Builder
    @JsonPOJOBuilder(withPrefix = "")
    public abstract static class Builder {
        public abstract Builder language(String language);
        public abstract Builder title(String title);
        public abstract Builder summary(String summary);
        @JsonProperty("desc")
        public abstract Builder description(String description);
        public abstract Builder news(String news);
        public abstract Builder closingLine(String closingLine);
        public abstract Builder playback(Playback playback);
        public abstract Builder download(Download download);
        public abstract Builder images(List<Media> images);
        public abstract Builder audio(List<Media> audio);
        public abstract Builder video(List<Media> video);
        public abstract Builder children(List<CompactMtgObject> children);
        public abstract Builder collections(List<CompactMtgObject> collections);
        public abstract Builder references(List<CompactMtgObject> references);
        public abstract Builder quiz(Quiz quiz);
        public abstract Builder childrenCount(Integer childrenCount);
        public abstract Builder audioDuration(Integer audioDuration);

        public abstract Content build();
    }
}