/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.List;

import travel.izi.api.model.enumeration.Category;
import travel.izi.api.model.enumeration.MtgObjectType;
import travel.izi.api.model.enumeration.Placement;
import travel.izi.api.model.enumeration.PlaybackType;
import travel.izi.api.model.enumeration.Status;

/**
 * MTG Object (Mobile Traveller Guide Object) is a core content object. MTGObject, depending on
 * its "type" property, represents any MTG Content object (museum, tour, navigation story,
 * tourist attraction, publisher, country or city).
 * MTGObject can be requested in full and compact formats.
 */
public abstract class MtgObject implements Parcelable {

    /**
     * Universally Unique Identifier of MtgObject.
     */
    @NonNull
    public abstract String uuid();

    /**
     * Type of MtgObject.
     */
    @NonNull
    public abstract MtgObjectType type();

    /**
     * Category of tour.
     */
    @Nullable
    public abstract Category category();

    /**
     * Type of playback algorithm. The field is returned for museum, collection and tour objects.
     */
    @Nullable
    public abstract PlaybackType playbackType();

    /**
     * Content publication status.
     */
    @NonNull
    public abstract Status status();

    /**
     * Publisher.
     */
    @Nullable
    public abstract CompactMtgObject publisher();

    /**
     * Content provider.
     */
    @Nullable
    public abstract ContentProvider contentProvider();

    /**
     * Array of all available languages of MtgObject content (ISO 639-1).
     * <p>EXCEPTION: For countries and cities is an array of languages
     * for which there are MTGObject in this locality.</p>
     */
    @Nullable
    public abstract List<String> languages();

    /**
     * Location of MtgObject.
     */
    @Nullable
    public abstract Location location();

    /**
     * Compact version of country.
     */
    @Nullable
    public abstract CompactMtgObject country();

    /**
     * Compact version of city.
     */
    @Nullable
    public abstract CompactMtgObject city();

    /**
     * Array of trigger zones.
     */
    @Nullable
    public abstract List<TriggerZone> triggerZones();

    /**
     * Object with mapping details.
     */
    @Nullable
    public abstract Map map();

    /**
     * Tour placement.
     */
    @Nullable
    public abstract Placement placement();

    /**
     * Distance of tour (in meters).
     */
    @Nullable
    public abstract Integer distance();

    /**
     * Total duration of tour (in seconds).
     * <p>NOTE: Value is calculated as: distance / category speed.</p>
     */
    @Nullable
    public abstract Integer duration();

    /**
     * Object with product purchasing details.
     */
    @Nullable
    public abstract Purchase purchase();

    /**
     * Available translations of the locality name (typical for city and country).
     */
    @Nullable
    public abstract List<Translation> translations();

    /**
     * Country code (ISO 3166-1) (typical for country).
     */
    @Nullable
    public abstract String countryCode();

    /**
     * The average of ratings/reviews. The section will be returned if the average was calculated.
     */
    @Nullable
    public abstract ReviewsResponse.Estimation reviews();

    /**
     * Tourist attraction map visibility. Can be: false – object shall be displayed on a map;
     * true – object is hidden from the user.
     */
    @Nullable
    public abstract Boolean hidden();

    /**
     * Typical for city. Defines if the city should be displayed on izi.TRAVEL site main page.
     */
    @Nullable
    public abstract Boolean visible();

    /**
     * SHA1 hash from "uuid + last updated" string.
     */
    @NonNull
    public abstract String hash();
}