/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.service;

import java.util.List;

import retrofit2.Call;
import retrofit2.http.GET;
import retrofit2.http.Query;
import travel.izi.api.model.entity.CompactMtgObject;

/**
 * Endpoints for Search.
 */
@SuppressWarnings("unused")
public interface SearchService {

    /**
     * Search all over IZI.directory content. Supports full-text search, area search,
     * transparent sorting and pagination.
     * <p>Examples:
     * <ul>
     * <li>
     * Find all cities which have content on English language and sort them by name.
     * <pre class="prettyprint">
     * /mtg/objects/search?languages=en&amp;type=city&amp;sort_by=title
     * </pre>
     * </li>
     * <li>
     * Find all tours, cities and countries which have content on English and Dutch and also have
     * `ams` in their title.
     * <pre class="prettyprint">
     * /mtg/objects/search?languages=en,nl&amp;type=city,country,tour&amp;query=ams
     * </pre>
     * </li>
     * <li>
     * Find all countries which have content on Russian and sort them alphabetically.
     * <pre class="prettyprint">
     * /mtg/objects/search?languages=ru&amp;type=country&amp;sort_by=title
     * </pre>
     * </li>
     * <li>
     * Find all tours with English language, do not include publisher section to results Compact
     * MtgObjects, but include city section.
     * <pre class="prettyprint">
     * /mtg/objects/search?languages=en&amp;type=tour&amp;includes=city&amp;except=publisher
     * </pre>
     * </li>
     * </ul>
     *
     * @param languages  Array of languages which should have MtgObject.
     * @param publishers Array of Publishers UUID’s. Defines a limit (filter) to make search in
     *                   certain Publishers.
     * @param includes   Array of {@link travel.izi.api.service.constant.ContentSectionModifier}
     *                   which should be included into MtgObject.
     * @param except     Array of {@link travel.izi.api.service.constant.ContentSectionModifier}
     *                   which should NOT be included into MtgObject.
     * @param types      List of requested types of objects, defaults are `tour`, `museum`.
     *                   Applicable values: `museum`, `tour`, `city`, `country`, `collection`,
     *                   `tourist_attraction`, `story_navigation` and `quest`. `quest` type
     *                   can be used to filter tours with playback type quest only, if the type
     *                   parameter will keep `tour` and `quest` values, the `quest` will be
     *                   ignored and tours with any playback types will be filtered.
     * @param query      Query for full-text search.
     * @param region     UUID of the city or country, limits search by city or country with
     *                   provided UUID.
     * @param lat_lon    Geo location used to sort found MtgObjects from provided point, defaults
     *                   to 0.0,0.0.
     * @param radius     Radius in meters, used only with #lat_lon param to search MtgObjects in
     *                   area.
     * @param ex_lat_lon Geo location, used only with #ex_radius to exclude from area search
     *                   certain area.
     * @param ex_radius  Radius in meters, used only with #ex_lat_lon to exclude from area search
     *                   certain area.
     * @param sort_by    Sorting field in format 'field:direction', currently possible values are
     *                   `rating`, `popularity` and `title`.
     *                   Example, "...&amp;sort_by=title:asc&amp;...". If results keeps `city` or
     *                   `country` objects in additional to `museum`, `tour`, etc, `country`
     *                   objects will be returned first, next `city` objects and finally rest of
     *                   objects. Each set will be sorted according to `sort_by` criteria.
     * @param cost       Filtering content by {@link travel.izi.api.service.constant.Cost}, by
     *                   default filter is not applied.
     * @param limit      Limit, defaults to 50.
     * @param offset     Offset, defaults to 0.
     * @param rating_min Filters content with rating equal or greater than `rating_min`, integer
     *                   values [0..10].
     * @param rating_max Filters content with rating equal or less than `rating_max`, integer
     *                   values [0..10].
     * @return List of various MtgObjects(Museum, Tour, City, Country, etc).
     * Integer, String, Integer, String, String, Long, Long, Boolean)
     */
    @GET("/mtg/objects/search?form=compact")
    Call<List<CompactMtgObject>> search(
            @Query("languages") String[] languages,
            @Query("publishers") String[] publishers,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("type") String[] types,
            @Query("query") String query,
            @Query("region") String region,
            @Query("lat_lon") String lat_lon,
            @Query("radius") Integer radius,
            @Query("ex_lat_lon") String ex_lat_lon,
            @Query("ex_radius") Integer ex_radius,
            @Query("sort_by") String sort_by,
            @Query("cost") String cost,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("rating_min") Integer rating_min,
            @Query("rating_max") Integer rating_max
    );

}