/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.service;

import retrofit.client.Response;
import retrofit.http.GET;
import retrofit.http.Path;
import retrofit.http.Query;
import travel.izi.api.IZITravelCallback;
import travel.izi.api.model.entity.CompactMtgObject;
import travel.izi.api.model.entity.FullMtgObject;
import travel.izi.api.model.entity.ProductIdResponse;
import travel.izi.api.model.enumeration.MtgObjectType;
import travel.izi.api.service.constant.ContentSectionModifier;

import java.util.List;

/**
 * Endpoints for MTG Object.
 */
@SuppressWarnings("unused")
public interface MtgObjectService {

    /**
     * Retrieve a full MtgObject by UUID param.
     *
     * @param uuid           UUID of MtgObject.
     * @param languages      Array of preferable content languages.
     * @param includes       Array of {@link ContentSectionModifier} (by default: {@link ContentSectionModifier#CHILDREN}).
     * @param except         The same array of {@link ContentSectionModifier} as above for excluding.
     * @param audio_duration Defines if the total audio duration shall be added to returned MTGObject(s).
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     *                       Field 'audio_duration’ will be added to root of compact form and to content section of full form.
     *                       The value of 'audio_duration’ is integer and keeps total duration in seconds of the object and children.
     * @param children_count Defines if the field `children_count` shall be added to full form of MTGObject in content section.
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     * @return Only one MtgObject is returned. The language of content is a first available
     * language from requested languages. If no available language or MtgObject with requested
     * UUID doesn't exist, 404 error code is returned.
     */
    @GET("/mtgobjects/{uuid}")
    List<FullMtgObject> getMtgObject(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Asynchronous version of {@link #getMtgObject(String, String[], String[], String[], Boolean, Boolean)}.
     */
    @GET("/mtgobjects/{uuid}")
    void getMTGObject(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/mtgobjects/{uuid}")
    Response getMTGObjectResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            @Query("form") String form
    );

    /**
     * Retrieve list of compact MtgObjects by set of UUIDs param.
     *
     * @param uuids          Set of MtgObject UUIDs separated by commas.
     * @param languages      Array of preferable content languages.
     * @param includes       Array of {@link ContentSectionModifier} (by default: {@link ContentSectionModifier#CHILDREN}).
     * @param except         The same array of {@link ContentSectionModifier} as above for excluding.
     * @param audio_duration Defines if the total audio duration shall be added to returned MTGObject(s).
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     *                       Field 'audio_duration’ will be added to root of compact form and to content section of full form.
     *                       The value of 'audio_duration’ is integer and keeps total duration in seconds of the object and children.
     * @return List of MtgObjects. The language of content is a first available
     * language from requested languages. If no available language or MtgObject with requested
     * UUIDs doesn't exist, 404 error code is returned.
     * @see #getFullMtgObjects(String, String[], String[], String[], Boolean, Boolean)
     */
    @GET("/mtgobjects/batch/{uuids}?form=compact")
    List<CompactMtgObject> getCompactMtgObjects(
            @Path("uuids") String uuids,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration
    );

    /**
     * Asynchronous version of {@link #getCompactMtgObjects(String, String[], String[], String[], Boolean)}.
     */
    @GET("/mtgobjects/batch/{uuids}?form=compact")
    void getCompactMTGObjects(
            @Path("uuids") String uuids,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            IZITravelCallback<List<CompactMtgObject>> callback
    );

    /**
     * Retrieve list of full MtgObjects by set of UUIDs param.
     *
     * @see #getCompactMtgObjects(String, String[], String[], String[], Boolean)
     */
    @GET("/mtgobjects/batch/{uuids}?form=full")
    List<FullMtgObject> getFullMtgObjects(
            @Path("uuids") String uuids,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Asynchronous version of {@link #getFullMtgObjects(String, String[], String[], String[], Boolean, Boolean)}.
     */
    @GET("/mtgobjects/batch/{uuids}?form=full")
    void getFullMTGObjects(
            @Path("uuids") String uuids,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/mtgobjects/batch/{uuids}")
    Response getMTGObjectsResponse(
            @Path("uuids") String uuids,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            @Query("form") String form
    );

    /**
     * Retrieve children of MtgObject in compact form.
     *
     * @param uuid           UUID of MtgObject.
     * @param languages      Array of preferable content languages.
     * @param includes       Array of {@link ContentSectionModifier}.
     * @param except         The same array of {@link ContentSectionModifier} as above for excluding.
     * @param limit          Number of results to return, up to 100 (50 by default)
     * @param offset         Get the results starting at a given offset (0 by default).
     * @param types          Array of requested types of children, i.e. for Museum children could be
     *                       {@link MtgObjectType#Exhibit} or {@link MtgObjectType#Collection},
     *                       while for Tour children could be {@link MtgObjectType#TouristAttraction}
     *                       or {@link MtgObjectType#StoryNavigation}.
     * @param show_hidden    If it set to true, then hidden tourist_attraction will be returned.
     * @param audio_duration Defines if the total audio duration shall be added to returned MTGObject(s).
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     *                       Field 'audio_duration’ will be added to root of compact form and to content section of full form.
     *                       The value of 'audio_duration’ is integer and keeps total duration in seconds of the object and children.
     * @return If MtgObject with requested UUID doesn't exist, 404 error code is returned.
     * @see #getFullChildren(String, String[], String[], String[], Long, Long, String[], Boolean, Boolean, Boolean)
     */
    @GET("/mtgobjects/{uuid}/children?form=compact")
    List<CompactMtgObject> getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("type") String[] types,
            @Query("show_hidden") Boolean show_hidden,
            @Query("audio_duration") Boolean audio_duration
    );

    /**
     * Asynchronous version of {@link #getCompactChildren(String, String[], String[], String[], Long, Long, String[], Boolean, Boolean)}.
     */
    @GET("/mtgobjects/{uuid}/children?form=compact")
    void getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("type") String[] types,
            @Query("show_hidden") Boolean show_hidden,
            @Query("audio_duration") Boolean audio_duration,
            IZITravelCallback<List<CompactMtgObject>> callback
    );

    /**
     * Retrieve children of MtgObject in full form.
     *
     * @see #getCompactChildren(String, String[], String[], String[], Long, Long, String[], Boolean, Boolean)
     */
    @GET("/mtgobjects/{uuid}/children?form=full")
    List<FullMtgObject> getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("type") String[] types,
            @Query("show_hidden") Boolean show_hidden,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Asynchronous version of {@link #getFullChildren(String, String[], String[], String[], Long, Long, String[], Boolean, Boolean, Boolean)}.
     */
    @GET("/mtgobjects/{uuid}/children?form=full")
    void getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("type") String[] types,
            @Query("show_hidden") Boolean show_hidden,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/mtgobjects/{uuid}/children")
    Response getChildrenResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("type") String[] types,
            @Query("show_hidden") Boolean show_hidden,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            @Query("form") String form
    );

    /**
     * Retrieve number of children of a MTGObject.
     *
     * @param uuid      UUID of MtgObject.
     * @param languages Array of preferable content languages.
     * @param types     Array of requested types of children, i.e. for Museum children could be
     *                  {@link MtgObjectType#Exhibit} or {@link MtgObjectType#Collection},
     *                  while for Tour children could be {@link MtgObjectType#TouristAttraction}
     *                  or {@link MtgObjectType#StoryNavigation}.
     * @return Amount of children of MTGObject with requested UUID as integer.
     */
    @GET("/mtgobjects/{uuid}/children/count")
    Integer getChildrenNumber(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("type") String[] types
    );

    /**
     * Asynchronous version of {@link #getChildrenNumber(String, String[], String[])}.
     */
    @GET("/mtgobjects/{uuid}/children/count")
    void getChildrenNumber(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("type") String[] types,
            IZITravelCallback<Integer> callback
    );

    @GET("/mtgobjects/{uuid}/children/count")
    Response getChildrenNumberResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("type") String[] types
    );

    /**
     * Retrieve a full museum object using its external IP address. IP address is detected by server if it is not set
     * directly via the parameter 'ip'.
     *
     * @param languages      Array of preferable content languages.
     * @param includes       Array of {@link ContentSectionModifier} (by default: {@link ContentSectionModifier#CHILDREN}).
     * @param except         The same array of {@link ContentSectionModifier} as above for excluding.
     * @param ip             External API value. Used to set external IP directly.
     * @param audio_duration Defines if the total audio duration shall be added to returned MTGObject(s).
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     *                       Field 'audio_duration’ will be added to root of compact form and to content section of full form.
     *                       The value of 'audio_duration’ is integer and keeps total duration in seconds of the object and children.
     * @param children_count Defines if the field `children_count` shall be added to full form of MTGObject in content section.
     *                       Values: `true` or `false`. Default value: `false`. Makes sense for MTGObjects museum, tour or collection.
     * @return Only one MtgObject is returned. The language of content is a first available
     * language from requested languages. If no available language or MtgObject with requested ip
     * doesn't exist, 404 error code is returned.
     */
    @GET("/mtgobjects/ip?form=full")
    List<FullMtgObject> getMuseumByIp(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("ip") String ip,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Asynchronous version of {@link #getMuseumByIp(String[], String[], String[], String, Boolean, Boolean)}.
     */
    @GET("/mtgobjects/ip?form=full")
    void getMuseumByIp(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("ip") String ip,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/mtgobjects/ip")
    Response getMuseumByIpResponse(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("ip") String ip,
            @Query("audio_duration") Boolean audio_duration,
            @Query("children_count_in_full_form") Boolean children_count,
            @Query("form") String form
    );

    /**
     * Retrieve product ID of MtgObject.
     *
     * @param uuid UUID of MtgObject.
     * @return If MtgObject with requested UUID doesn't exist, 404 error code is returned.
     */
    @GET("/mtgobjects/{uuid}/product_id")
    ProductIdResponse getProductId(
            @Path("uuid") String uuid
    );

    /**
     * Asynchronous version of {@link #getProductId(String)}.
     */
    @GET("/mtgobjects/{uuid}/product_id")
    void getProductId(
            @Path("uuid") String uuid,
            IZITravelCallback<ProductIdResponse> callback
    );

    @GET("/mtgobjects/{uuid}/product_id")
    Response getProductIdResponse(
            @Path("uuid") String uuid
    );

    /**
     * Search for museums and/or tours by product IDs.
     *
     * @param languages  Array of preferable languages.
     * @param productIds Array of product IDs.
     * @param includes   Array of {@link ContentSectionModifier}.
     * @param except     The same array of {@link ContentSectionModifier} as above for excluding.
     * @return If MtgObject with requested product IDs doesn't exist, 404 error code is returned.
     * @see #searchFullByProductIds(String[], String[], String[], String[])
     */
    @GET("/mtgobjects/search/paid?form=compact")
    List<CompactMtgObject> searchCompactByProductIds(
            @Query("languages") String[] languages,
            @Query("product_ids") String[] productIds,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Asynchronous version of {@link #searchCompactByProductIds(String[], String[], String[], String[])}.
     */
    @GET("/mtgobjects/search/paid?form=compact")
    void searchCompactByProductIds(
            @Query("languages") String[] languages,
            @Query("product_ids") String[] productIds,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            IZITravelCallback<List<CompactMtgObject>> callback
    );

    /**
     * Search for museums and/or tours by product IDs.
     *
     * @see #searchCompactByProductIds(String[], String[], String[], String[])
     */
    @GET("/mtgobjects/search/paid?form=full")
    List<FullMtgObject> searchFullByProductIds(
            @Query("languages") String[] languages,
            @Query("product_ids") String[] productIds,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Asynchronous version of {@link #searchFullByProductIds(String[], String[], String[], String[])}.
     */
    @GET("/mtgobjects/search/paid?form=full")
    void searchFullByProductIds(
            @Query("languages") String[] languages,
            @Query("product_ids") String[] productIds,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/mtgobjects/search/paid")
    Response searchByProductIdsResponse(
            @Query("languages") String[] languages,
            @Query("product_ids") String[] productIds,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("form") String form
    );

}