/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.service;

import retrofit.client.Response;
import retrofit.http.*;
import travel.izi.api.IZITravelCallback;
import travel.izi.api.model.entity.Review;
import travel.izi.api.model.entity.ReviewsResponse;

/**
 * Endpoints for Reviews.
 */
@SuppressWarnings("unused")
public interface ReviewService {

    /**
     * Get rating and reviews for MtgObject by UUID.
     *
     * @param uuid     UUID of the MtgObject.
     * @param language Returns records for certain language. Example: "&amp;lang=ru". Skip argument for all languages.
     * @param limit    The number of individual records that are returned in each page.
     *                 Default 25, min:0 (only metadata be returned), max:100.
     * @param offset   This offsets the start of each page by the number of records specified.
     *                 Default 0.
     * @return ReviewsResponse.
     */
    @GET("/mtgobjects/{uuid}/reviews")
    ReviewsResponse getReviews(
            @Path("uuid") String uuid,
            @Query("lang") String language,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Asynchronous version of {@link #getReviews(String, String, Long, Long)}.
     */
    @GET("/mtgobjects/{uuid}/reviews")
    void getReviews(
            @Path("uuid") String uuid,
            @Query("lang") String language,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            IZITravelCallback<ReviewsResponse> callback
    );

    @GET("/mtgobjects/{uuid}/reviews")
    Response getReviewsResponse(
            @Path("uuid") String uuid,
            @Query("lang") String language,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Add rating or/and review to a MtgObject.
     *
     * @param uuid   UUID of the MtgObject.
     * @param review A rating/review.
     * @return ReviewsResponse only with an estimation and created review item. If it's first review, then estimation is null.
     */
    @POST("/mtgobjects/{uuid}/reviews")
    ReviewsResponse addReview(
            @Path("uuid") String uuid,
            @Body Review review
    );

    /**
     * Asynchronous version of {@link #addReview(String, Review)}.
     */
    @POST("/mtgobjects/{uuid}/reviews")
    void addReview(
            @Path("uuid") String uuid,
            @Body Review review,
            IZITravelCallback<ReviewsResponse> callback
    );

    @POST("/mtgobjects/{uuid}/reviews")
    Response addReviewResponse(
            @Path("uuid") String uuid,
            @Body Review review
    );

}