/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api;

import com.google.gson.annotations.SerializedName;
import retrofit.RetrofitError;
import retrofit.RetrofitError.Kind;
import retrofit.client.Response;

import java.io.Serializable;

@SuppressWarnings("unused")
public class IZITravelError extends RuntimeException {

    /**
     * The HTTP response error.
     */
    public static class HttpError implements Serializable {
        private static final long serialVersionUID = 1942575303158771456L;

        /**
         * The error code.
         */
        public int code;

        /**
         * The error message.
         */
        @SerializedName("error")
        public String message;

        @Override
        public String toString() {
            return code + " " + message;
        }

    }

    private final String url;
    private final Response response;
    private final HttpError httpError;
    private final Kind kind;

    public static IZITravelError from(RetrofitError error) {
        HttpError httpError = (HttpError) error.getBodyAs(HttpError.class);
        String message = httpError != null ? httpError.toString() : error.getMessage();
        return new IZITravelError(message, error.getUrl(), error.getResponse(), httpError, error.getKind(), error.getCause());
    }

    private IZITravelError(String message, String url, Response response, HttpError httpError, Kind kind, Throwable exception) {
        super(message, exception);
        this.url = url;
        this.response = response;
        this.httpError = httpError;
        this.kind = kind;
    }

    /**
     * The request URL which produced the error.
     */
    public String getUrl() {
        return url;
    }

    /**
     * Response object containing status code, headers, body, etc.
     */
    public Response getResponse() {
        return response;
    }

    /**
     * The HTTP response error.
     */
    public HttpError getHttpError() {
        return httpError;
    }

    /**
     * The event kind which triggered this error.
     */
    public Kind getKind() {
        return kind;
    }

}