/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.service;

import retrofit.client.Response;
import retrofit.http.GET;
import retrofit.http.Path;
import retrofit.http.Query;
import travel.izi.api.IZITravelCallback;
import travel.izi.api.model.entity.CompactMtgObject;
import travel.izi.api.model.entity.FullMtgObject;

import java.util.List;

/**
 * Endpoints for City.
 */
@SuppressWarnings("unused")
public interface CityService {

    /**
     * Get list of compact City objects which have content on requested languages.
     *
     * @param languages Array of languages which should have city.
     * @param includes  Array of sections which should be included to city object.
     * @param except    Array of sections which should NOT be included to city object.
     * @param visible   If it equals to 'true', then returns cities marked as visible only.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return City objects which have content on requested languages. Returns empty list of there
     * are no matched cities.
     * @see #getFullCities(String[], String[], String[], Boolean, Long, Long)
     */
    @GET("/cities?form=compact")
    List<CompactMtgObject> getCompactCities(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("visible") Boolean visible,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Asynchronous version of {@link #getCompactCities(String[], String[], String[], Boolean, Long, Long)}.
     */
    @GET("/cities?form=compact")
    void getCompactCities(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("visible") Boolean visible,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            IZITravelCallback<List<CompactMtgObject>> callback
    );

    /**
     * Get list of full City objects which have content on requested languages.
     *
     * @see #getCompactCities(String[], String[], String[], Boolean, Long, Long)
     */
    @GET("/cities?form=full")
    List<FullMtgObject> getFullCities(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("visible") Boolean visible,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Asynchronous version of {@link #getFullCities(String[], String[], String[], Boolean, Long, Long)}.
     */
    @GET("/cities?form=full")
    void getFullCities(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("visible") Boolean visible,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/cities")
    Response getCitiesResponse(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("visible") Boolean visible,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("form") String form
    );

    /**
     * Get full City object by UUID.
     *
     * @param uuid      UUID of the City.
     * @param languages Array of languages which should have city.
     * @param includes  Array of sections which should be included to city object.
     * @param except    Array of sections which should NOT be included to city object.
     * @return City object or 404 HTTP code if there is no City with provided UUID.
     */
    @GET("/cities/{uuid}")
    FullMtgObject getCity(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Asynchronous version of {@link #getCity(String, String[], String[], String[])}.
     */
    @GET("/cities/{uuid}")
    void getCity(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            IZITravelCallback<FullMtgObject> callback
    );

    @GET("/cities/{uuid}")
    Response getCityResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("form") String form
    );

    /**
     * Get City children in compact form.
     *
     * @param uuid      UUID of the City.
     * @param languages Array of requested languages, i.e. City's children should have one of the requested languages.
     * @param includes  Array of sections which should be included into response.
     * @param except    Array of sections which should NOT be included into response.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return Tours and Museums which belongs to the city. Returns 404 HTTP code if there is no City with provided UUID.
     * @see #getFullChildren(String, String[], String[], String[], Long, Long, Boolean)
     */
    @GET("/cities/{uuid}/children?form=compact")
    List<CompactMtgObject> getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Asynchronous version of {@link #getCompactChildren(String, String[], String[], String[], Long, Long)}.
     */
    @GET("/cities/{uuid}/children?form=compact")
    void getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            IZITravelCallback<List<CompactMtgObject>> callback
    );

    /**
     * Get City children in full form.
     *
     * @see #getCompactChildren(String, String[], String[], String[], Long, Long)
     */
    @GET("/cities/{uuid}/children?form=full")
    List<FullMtgObject> getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Asynchronous version of {@link #getFullChildren(String, String[], String[], String[], Long, Long, Boolean)}.
     */
    @GET("/cities/{uuid}/children?form=full")
    void getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("children_count_in_full_form") Boolean children_count,
            IZITravelCallback<List<FullMtgObject>> callback
    );

    @GET("/cities/{uuid}/children")
    Response getChildrenResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("children_count_in_full_form") Boolean children_count,
            @Query("form") String form
    );

    /**
     * Get the number of City’s children (museums, tours).
     *
     * @param uuid      UUID of the City.
     * @param languages Array of preferable content languages.
     * @return Number of children of requested City on requested languages (returned as an integer).
     */
    @GET("/cities/{uuid}/children/count")
    Integer getChildrenNumber(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages
    );

    /**
     * Asynchronous version of {@link #getChildrenNumber(String, String[])}.
     */
    @GET("/cities/{uuid}/children/count")
    void getChildrenNumber(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            IZITravelCallback<Integer> callback
    );

    @GET("/cities/{uuid}/children/count")
    Response getChildrenNumberResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages
    );

    /**
     * Get full Country object by City.
     *
     * @param uuid      UUID of the City.
     * @param languages Array of languages which should have city.
     * @param includes  Array of sections which should be included to country object.
     * @param except    Array of sections which should NOT be included to country object.
     * @return Country object or 404 HTTP code if there is no City with provided UUID or City does not have Country.
     */
    @GET("/cities/{uuid}/country")
    FullMtgObject getCountry(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Asynchronous version of {@link #getCountry(String, String[], String[], String[])}.
     */
    @GET("/cities/{uuid}/country")
    void getCountry(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            IZITravelCallback<FullMtgObject> callback
    );

    @GET("/cities/{uuid}/country")
    Response getCountryResponse(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("form") String form
    );

}