/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api;

import com.google.gson.GsonBuilder;
import com.squareup.okhttp.Cache;
import com.squareup.okhttp.OkHttpClient;
import retrofit.ErrorHandler;
import retrofit.RequestInterceptor;
import retrofit.RestAdapter;
import retrofit.RetrofitError;
import retrofit.client.OkClient;
import retrofit.converter.GsonConverter;
import travel.izi.api.service.*;
import travel.izi.api.util.GsonHelper;
import travel.izi.api.util.MediaHelper;

import java.io.File;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("unused")
public class IZITravel {

    public static final Server SERVER_PRODUCTION = new Server(
            "http://api.izi.travel",
            "http://media.izi.travel"
    );
    public static final Server SERVER_STAGING = new Server(
            "http://api.stage.izi.travel",
            "http://media.stage.izi.travel"
    );
    public static final Server SERVER_DEVELOPMENT = new Server(
            "http://api.dev.izi.travel",
            "http://media.dev.izi.travel"
    );

    /**
     * The default API version.
     */
    public static final String API_VERSION = "1.2.4";

    /**
     * The default connect timeout for new connections.
     */
    private static final int CONNECT_TIMEOUT_MILLIS = 15 * 1000; // 15s

    /**
     * The default read timeout for new connections.
     */
    private static final int READ_TIMEOUT_MILLIS = 20 * 1000; // 20s

    private final String mApiKey;
    private final String mApiVersion;
    private final String mApiPassword;
    private final String mUserAgent;
    private final Server mServer;
    private final Cache mCache;
    private final boolean mLoggingEnabled;

    private final RestAdapter mRestAdapter;

    private IZITravel(String apiKey, String apiVersion, String apiPassword,
            String userAgent, Server server, Cache cache, boolean loggingEnabled) {
        mApiKey = apiKey;
        mApiVersion = apiVersion;
        mApiPassword = apiPassword;
        mUserAgent = userAgent;
        mServer = server;
        mCache = cache;
        mLoggingEnabled = loggingEnabled;

        mRestAdapter = buildRestAdapter();
    }

    protected RestAdapter buildRestAdapter() {
        RestAdapter.Builder builder = new RestAdapter.Builder()
                .setEndpoint(mServer.getApiUrl())
                .setClient(new OkClient((getOkHttpClient())))
                .setConverter(new GsonConverter(getGsonBuilder().create()))
                .setErrorHandler(getErrorHandler())
                .setRequestInterceptor(getRequestInterceptor());
        if (mLoggingEnabled) {
            builder.setLogLevel(RestAdapter.LogLevel.FULL);
        }
        return builder.build();
    }

    protected OkHttpClient getOkHttpClient() {
        OkHttpClient client = new OkHttpClient();
        client.setConnectTimeout(CONNECT_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
        client.setReadTimeout(READ_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
        if (mCache != null) {
            client.setCache(mCache);
        }
        return client;
    }

    protected GsonBuilder getGsonBuilder() {
        return GsonHelper.getGsonBuilder();
    }

    protected ErrorHandler getErrorHandler() {
        return new ErrorHandler() {
            @Override
            public Throwable handleError(RetrofitError error) {
                return IZITravelError.from(error);
            }
        };
    }

    protected RequestInterceptor getRequestInterceptor() {
        return new RequestInterceptor() {
            @Override
            public void intercept(RequestFacade request) {
                request.addHeader("Accept", String.format("application/izi-api-v%s+json", mApiVersion));
                request.addHeader("X-IZI-API-KEY", mApiKey);
                if (mApiPassword != null) {
                    request.addHeader("X-IZI-API-PASSWORD", mApiPassword);
                }
                if (mUserAgent != null) {
                    request.addHeader("User-Agent", mUserAgent);
                }
                request.addHeader("Cache-Control", "max-stale=3600");
            }
        };
    }

    public CityService cityService() {
        return mRestAdapter.create(CityService.class);
    }

    public CountryService countryService() {
        return mRestAdapter.create(CountryService.class);
    }

    public FeaturedService featuredService() {
        return mRestAdapter.create(FeaturedService.class);
    }

    public LanguageService languageService() {
        return mRestAdapter.create(LanguageService.class);
    }

    public MtgObjectService mtgObjectService() {
        return mRestAdapter.create(MtgObjectService.class);
    }

    public PublisherService publisherService() {
        return mRestAdapter.create(PublisherService.class);
    }

    public ReviewService reviewService() {
        return mRestAdapter.create(ReviewService.class);
    }

    public SearchService searchService() {
        return mRestAdapter.create(SearchService.class);
    }

    public MediaHelper mediaHelper() {
        return new MediaHelper(mServer.getMediaUrl());
    }

    /**
     * The default {@link IZITravel} instance.
     * <p>This instance is automatically initialized with defaults that are suitable to most
     * implementations.</p>
     * <ul>
     * <li>API version is {@link #API_VERSION}</li>
     * <li>Server is {@link #SERVER_PRODUCTION}</li>
     * </ul>
     * <p>If these settings do not meet the requirements of your application you can construct your own
     * with full control over the configuration by using {@link IZITravel.Builder} to create a
     * {@link IZITravel} instance.</p>
     */
    public static IZITravel withApiKey(String apiKey) {
        return new Builder().apiKey(apiKey).build();
    }

    /**
     * A helper class for creating {@link IZITravel} instances.
     */
    public static class Builder {
        private String mApiKey;
        private String mApiVersion;
        private String mApiPassword;
        private String mUserAgent;
        private Server mServer;
        private Cache mCache;

        private boolean mLoggingEnabled;

        /**
         * Set API key. All API methods require a valid API key.
         */
        public Builder apiKey(String apiKey) {
            if (apiKey == null) {
                throw new IllegalArgumentException("ApiKey must not be null.");
            }
            mApiKey = apiKey;
            return this;
        }

        /**
         * Set API version. By default used {@link #API_VERSION}.
         */
        public Builder apiVersion(String apiVersion) {
            if (apiVersion == null) {
                throw new IllegalArgumentException("ApiVersion must not be null.");
            }
            mApiVersion = apiVersion;
            return this;
        }

        /**
         * Set API password. Using for getting access to "limited" content. By default available only
         * public content.
         */
        public Builder apiPassword(String apiPassword) {
            mApiPassword = apiPassword;
            return this;
        }

        /**
         * Set user agent. By default user agent is null.
         */
        public Builder userAgent(String userAgent) {
            mUserAgent = userAgent;
            return this;
        }

        /**
         * Set {@link Server}. By default using {@link #SERVER_PRODUCTION}.
         */
        public Builder server(Server server) {
            if (server == null) {
                throw new IllegalArgumentException("Server must not be null.");
            }
            mServer = server;
            return this;
        }

        /**
         * Set HTTP response cache.
         *
         * @param cacheDirectory Cache directory.
         * @param cacheSize      Cache size in bytes.
         */
        public Builder cache(File cacheDirectory, int cacheSize) {
            mCache = new Cache(cacheDirectory, cacheSize);
            return this;
        }

        /**
         * Whether to return more detailed log output. By default logging is disabled.
         */
        public Builder loggingEnabled(boolean enabled) {
            mLoggingEnabled = enabled;
            return this;
        }

        /**
         * Create the {@link IZITravel} instance.
         */
        public IZITravel build() {
            if (mApiKey == null) {
                throw new IllegalArgumentException("API key is not specified");
            }
            if (mApiVersion == null) {
                mApiVersion = API_VERSION;
            }
            if (mServer == null) {
                mServer = SERVER_PRODUCTION;
            }
            return new IZITravel(mApiKey, mApiVersion, mApiPassword, mUserAgent, mServer, mCache, mLoggingEnabled);
        }
    }

    /**
     * The server.
     */
    public static class Server {
        private final String apiUrl;
        private final String mediaUrl;

        public Server(String apiUrl, String mediaUrl) {
            this.apiUrl = apiUrl;
            this.mediaUrl = mediaUrl;
        }

        public String getApiUrl() {
            return apiUrl;
        }

        public String getMediaUrl() {
            return mediaUrl;
        }
    }

}