/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.service;

import retrofit.http.GET;
import retrofit.http.Path;
import retrofit.http.Query;
import travel.izi.api.model.entity.CompactMtgObject;
import travel.izi.api.model.entity.FullMtgObject;

import java.util.List;

/**
 * Endpoints for Country.
 */
@SuppressWarnings("unused")
public interface CountryService {

    /**
     * Get list of compact Country objects which have content on requested languages.
     *
     * @param languages Array of languages which should have country.
     * @param includes  Array of sections which should be included to country object.
     * @param except    Array of sections which should NOT be included to country object.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return List of countries which have content on requested languages. Returns empty list of there
     * are no matched countries.
     * @see #getFullCountries(String[], String[], String[], Long, Long)
     */
    @GET("/countries?form=compact")
    List<CompactMtgObject> getCompactCountries(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Get list of full Country objects which have content on requested languages.
     *
     * @see #getCompactCountries(String[], String[], String[], Long, Long)
     */
    @GET("/countries?form=full")
    List<FullMtgObject> getFullCountries(
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Get full Country object by UUID.
     *
     * @param uuid      UUID of the Country.
     * @param languages Array of languages which should have country.
     * @param includes  Array of sections which should be included to country object.
     * @param except    Array of sections which should NOT be included to country object.
     * @return Country or 404 HTTP code if there is no Country with provided UUID.
     */
    @GET("/countries/{uuid}")
    FullMtgObject getCountry(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except
    );

    /**
     * Get Country children in compact form.
     *
     * @param uuid      UUID of the Country.
     * @param languages Array of requested languages, i.e. Country's children should have one of the requested languages.
     * @param includes  Array of sections which should be included into response.
     * @param except    Array of sections which should NOT be included into response.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return Tours and Museums which belongs to the country. Returns 404 HTTP code if there is no Country with provided UUID.
     * @see #getFullChildren(String, String[], String[], String[], Long, Long, Boolean)
     */
    @GET("/countries/{uuid}/children?form=compact")
    List<CompactMtgObject> getCompactChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Get Country children in full form.
     *
     * @see #getCompactChildren(String, String[], String[], String[], Long, Long)
     */
    @GET("/countries/{uuid}/children?form=full")
    List<FullMtgObject> getFullChildren(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset,
            @Query("children_count_in_full_form") Boolean children_count
    );

    /**
     * Get list of compact City objects which belong to the Country with requested UUID.
     *
     * @param uuid      UUID of the Country.
     * @param languages Array of requested languages, each Country's City should have content at one of the requested language.
     * @param includes  Array of sections which should be included to city object.
     * @param except    Array of sections which should NOT be included to city object.
     * @param limit     Limit for pagination, defaults to 20.
     * @param offset    Offset for pagination, defaults to 0.
     * @return List of cities which belong to requested Country and have content on requested languages object or
     * 404 HTTP code if there is no Country with provided UUID.
     * @see #getFullCities(String, String[], String[], String[], Long, Long)
     */
    @GET("/countries/{uuid}/cities?form=compact")
    List<CompactMtgObject> getCompactCities(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

    /**
     * Get list of full City objects which belong to the Country with requested UUID.
     *
     * @see #getCompactCities(String, String[], String[], String[], Long, Long)
     */
    @GET("/countries/{uuid}/cities?form=full")
    List<FullMtgObject> getFullCities(
            @Path("uuid") String uuid,
            @Query("languages") String[] languages,
            @Query("includes") String[] includes,
            @Query("except") String[] except,
            @Query("limit") Long limit,
            @Query("offset") Long offset
    );

}