/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import java.util.List;

import auto.parcel.AutoParcel;
import travel.izi.api.model.enumeration.Category;
import travel.izi.api.model.enumeration.MtgObjectType;
import travel.izi.api.model.enumeration.Placement;
import travel.izi.api.model.enumeration.PlaybackType;
import travel.izi.api.model.enumeration.Status;

/**
 * The compact version of {@link MtgObject}. Used to display a list of items.
 */
@AutoParcel
@JsonDeserialize(builder = AutoParcel_CompactMtgObject.Builder.class)
public abstract class CompactMtgObject extends MtgObject {

    public static Builder builder() {
        return new AutoParcel_CompactMtgObject.Builder();
    }

    /**
     * Language (ISO 639-1) of object's content.
     */
    @NonNull
    public abstract String language();

    /**
     * Title.
     */
    @NonNull
    public abstract String title();

    /**
     * Shortened description.
     */
    @NonNull
    public abstract String summary();

    /**
     * Full description. Only for objects from "children" section.
     */
    @Nullable
    @JsonProperty("desc")
    public abstract String description();

    /**
     * Array of images data.
     * <p>NOTE: Contains only one object (first image).</p>
     */
    @Nullable
    public abstract List<Media> images();

    /**
     * Quiz data. Only for objects from "children" section.
     */
    @Nullable
    public abstract Quiz quiz();

    /**
     * Route in KML format (typical for tour). This route is for display on map purposes only.
     */
    @Nullable
    public abstract String route();

    /**
     * Number of children.
     */
    @Nullable
    public abstract Integer childrenCount();

    @AutoParcel.Builder
    @JsonPOJOBuilder(withPrefix = "")
    public abstract static class Builder {
        public abstract Builder uuid(String uuid);
        public abstract Builder type(MtgObjectType type);
        public abstract Builder category(Category category);
        public abstract Builder playbackType(PlaybackType playbackType);
        public abstract Builder status(Status status);
        public abstract Builder publisher(CompactMtgObject publisher);
        public abstract Builder contentProvider(ContentProvider contentProvider);
        public abstract Builder languages(List<String> languages);
        public abstract Builder location(Location location);
        public abstract Builder country(CompactMtgObject country);
        public abstract Builder city(CompactMtgObject city);
        public abstract Builder triggerZones(List<TriggerZone> triggerZones);
        public abstract Builder map(Map map);
        public abstract Builder placement(Placement placement);
        public abstract Builder distance(Integer distance);
        public abstract Builder duration(Integer duration);
        public abstract Builder purchase(Purchase purchase);
        public abstract Builder translations(List<Translation> translations);
        public abstract Builder countryCode(String countryCode);
        public abstract Builder reviews(ReviewsResponse.Estimation reviews);
        public abstract Builder hidden(Boolean hidden);
        public abstract Builder visible(Boolean visible);
        public abstract Builder hash(String hash);

        public abstract Builder language(String language);
        public abstract Builder title(String title);
        public abstract Builder summary(String summary);
        @JsonProperty("desc")
        public abstract Builder description(String description);
        public abstract Builder images(List<Media> images);
        public abstract Builder quiz(Quiz quiz);
        public abstract Builder route(String route);
        public abstract Builder childrenCount(Integer childrenCount);

        public abstract CompactMtgObject build();
    }
}