/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import java.util.List;

import auto.parcel.AutoParcel;
import travel.izi.api.model.enumeration.Category;
import travel.izi.api.model.enumeration.MtgObjectType;
import travel.izi.api.model.enumeration.Placement;
import travel.izi.api.model.enumeration.PlaybackType;
import travel.izi.api.model.enumeration.Status;

/**
 * The full version of {@link MtgObject}. Used to display details of particular content object.
 */
@AutoParcel
@JsonDeserialize(builder = AutoParcel_FullMtgObject.Builder.class)
public abstract class FullMtgObject extends MtgObject {

    public static Builder builder() {
        return new AutoParcel_FullMtgObject.Builder();
    }

    /**
     * Array of language specific MtgObject content: texts, descriptions, media. Everything
     * displayed to end-user come within this array.
     */
    @NonNull
    @JsonProperty("content")
    public abstract List<Content> contents();

    /**
     * The UUID of parent MtgObject. For example, tour’s uuid of a tourist attraction,
     * museum’s uuid of an exhibit.
     */
    @Nullable
    @Deprecated
    public abstract String parentUuid();

    /**
     * Schedule details (typical for museum).
     */
    @Nullable
    public abstract Schedule schedule();

    /**
     * Contacts (typical for museum/publisher).
     */
    @Nullable
    public abstract Contacts contacts();

    /**
     * Sponsors (typical for museum/tour).
     */
    @Nullable
    public abstract List<Sponsor> sponsors();

    /**
     * The size of the media content (audio + video + images) for the object and all its children.
     * Typical for museum/tour/collection.
     */
    @Nullable
    public abstract Long size();

    @AutoParcel.Builder
    @JsonPOJOBuilder(withPrefix = "")
    public abstract static class Builder {
        public abstract Builder uuid(String uuid);
        public abstract Builder type(MtgObjectType type);
        public abstract Builder category(Category category);
        public abstract Builder playbackType(PlaybackType playbackType);
        public abstract Builder status(Status status);
        public abstract Builder publisher(CompactMtgObject publisher);
        public abstract Builder contentProvider(ContentProvider contentProvider);
        public abstract Builder languages(List<String> languages);
        public abstract Builder location(Location location);
        public abstract Builder country(CompactMtgObject country);
        public abstract Builder city(CompactMtgObject city);
        public abstract Builder triggerZones(List<TriggerZone> triggerZones);
        public abstract Builder map(Map map);
        public abstract Builder placement(Placement placement);
        public abstract Builder distance(Integer distance);
        public abstract Builder duration(Integer duration);
        public abstract Builder purchase(Purchase purchase);
        public abstract Builder translations(List<Translation> translations);
        public abstract Builder countryCode(String countryCode);
        public abstract Builder reviews(ReviewsResponse.Estimation reviews);
        public abstract Builder hidden(Boolean hidden);
        public abstract Builder visible(Boolean visible);
        public abstract Builder hash(String hash);

        @JsonProperty("content")
        public abstract Builder contents(List<Content> contents);
        public abstract Builder parentUuid(String parentUuid);
        public abstract Builder schedule(Schedule schedule);
        public abstract Builder contacts(Contacts contacts);
        public abstract Builder sponsors(List<Sponsor> sponsors);
        public abstract Builder size(Long size);

        public abstract FullMtgObject build();
    }
}