/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import java.util.List;

import auto.parcel.AutoParcel;
import travel.izi.api.model.enumeration.TriggerZoneConditionType;

@AutoParcel
@JsonDeserialize(builder = AutoParcel_QuestSegment.Builder.class)
public abstract class QuestSegment implements Parcelable {

    public static Builder builder() {
        return new AutoParcel_QuestSegment.Builder();
    }

    /**
     * Unique segment number, the numbering started from 1. First segment in Quests is always
     * opened segment (visible to the end-user).
     */
    @NonNull
    public abstract Integer number();

    /**
     * An array of children UUIDs assigned to the segment (ordered).
     */
    @NonNull
    public abstract List<String> items();

    /**
     * Route of segment in KML format. Typical for tour.
     */
    @Nullable
    public abstract String route();

    /**
     * Defines conditions of segment opening via operators 'or' or 'and'.
     */
    @Nullable
    @JsonProperty("open_segment_when")
    public abstract ConditionsSet openSegmentConditions();

    /**
     * Defines conditions of quest ending via operators 'or' or 'and'.
     */
    @Nullable
    @JsonProperty("end_quest_when")
    public abstract ConditionsSet endQuestConditions();

    @AutoParcel.Builder
    @JsonPOJOBuilder(withPrefix = "")
    public abstract static class Builder {
        public abstract Builder number(Integer number);
        public abstract Builder items(List<String> items);
        public abstract Builder route(String route);
        @JsonProperty("open_segment_when")
        public abstract Builder openSegmentConditions(ConditionsSet openSegmentConditions);
        @JsonProperty("end_quest_when")
        public abstract Builder endQuestConditions(ConditionsSet endQuestConditions);

        public abstract QuestSegment build();
    }

    @AutoParcel
    @JsonDeserialize(builder = AutoParcel_QuestSegment_ConditionsSet.Builder.class)
    public abstract static class ConditionsSet implements Parcelable {

        public static Builder builder() {
            return new AutoParcel_QuestSegment_ConditionsSet.Builder();
        }

        /**
         * Array of conditions. Boolean expression is true if one of condition is true.
         */
        @Nullable
        public abstract List<ConditionStub> or();

        /**
         * Array of conditions. Boolean expression is true if all of conditions are true.
         */
        @Nullable
        public abstract List<ConditionStub> and();

        @AutoParcel.Builder
        @JsonPOJOBuilder(withPrefix = "")
        public abstract static class Builder {
            public abstract Builder or(List<ConditionStub> or);
            public abstract Builder and(List<ConditionStub> and);

            public abstract ConditionsSet build();
        }
    }

    @AutoParcel
    @JsonDeserialize(builder = AutoParcel_QuestSegment_ConditionStub.Builder.class)
    public abstract static class ConditionStub implements Parcelable {
        public static Builder builder() {
            return new AutoParcel_QuestSegment_ConditionStub.Builder();
        }

        @Nullable
        public abstract QuizCondition quizCondition();

        @Nullable
        public abstract TriggerZoneCondition triggerZoneCondition();

        @AutoParcel.Builder
        @JsonPOJOBuilder(withPrefix = "")
        public abstract static class Builder {
            public abstract Builder quizCondition(QuizCondition quizCondition);
            public abstract Builder triggerZoneCondition(TriggerZoneCondition triggerZoneCondition);

            public abstract ConditionStub build();
        }
    }

    public static abstract class Condition implements Parcelable {

        /**
         * UUID of object.
         */
        @NonNull
        public abstract String item();
    }

    @AutoParcel
    @JsonDeserialize(builder = AutoParcel_QuestSegment_QuizCondition.Builder.class)
    public abstract static class QuizCondition extends Condition {

        public static Builder builder() {
            return new AutoParcel_QuestSegment_QuizCondition.Builder();
        }

        /**
         * Defines if quiz correctly answered.
         */
        @NonNull
        public abstract Boolean correctAnswered();

        @AutoParcel.Builder
        @JsonPOJOBuilder(withPrefix = "")
        public abstract static class Builder {
            public abstract Builder item(String item);
            public abstract Builder correctAnswered(Boolean correctAnswered);

            abstract QuizCondition autoBuild();

            public QuizCondition build() {
                try {
                    return autoBuild();
                } catch (IllegalStateException e) {
                    return correctAnswered(true).autoBuild();
                }
            }
        }
    }

    @AutoParcel
    @JsonDeserialize(builder = AutoParcel_QuestSegment_TriggerZoneCondition.Builder.class)
    public abstract static class TriggerZoneCondition extends Condition {

        public static Builder builder() {
            return new AutoParcel_QuestSegment_TriggerZoneCondition.Builder();
        }

        /**
         * Defines the type of trigger zone condition.
         */
        @NonNull
        public abstract TriggerZoneConditionType type();

        @AutoParcel.Builder
        @JsonPOJOBuilder(withPrefix = "")
        public abstract static class Builder {
            public abstract Builder item(String item);
            public abstract Builder type(TriggerZoneConditionType type);

            public abstract TriggerZoneCondition build();
        }
    }
}