/*
 * Copyright (C) 2015 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.api.model.entity;

import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import java.util.Date;
import java.util.List;

import auto.parcel.AutoParcel;

/**
 * Response object for endpoint {@link travel.izi.api.service.ReviewService#reviews(String,
 * String, Long, Long)}.
 */
@AutoParcel
public abstract class ReviewsResponse implements Parcelable {

    @JsonCreator
    public static ReviewsResponse create(@JsonProperty("metadata") Estimation estimation,
                                         @JsonProperty("data") List<Review> reviews,
                                         @JsonProperty("paging") Paging paging) {
        return new AutoParcel_ReviewsResponse(estimation, reviews, paging);
    }

    @NonNull
    @JsonProperty("metadata")
    public abstract Estimation estimation();

    @Nullable
    @JsonProperty("data")
    public abstract List<Review> reviews();

    @NonNull
    public abstract Paging paging();

    /**
     * The average of ratings/reviews. If the average is not calculated yet, the section will be
     * returned
     * where 'ratingAverage','ratingsCount', 'reviewsCount' fields will be 0 and 'date' will be
     * current time.
     */
    @AutoParcel
    @JsonDeserialize(builder = AutoParcel_ReviewsResponse_Estimation.Builder.class)
    public abstract static class Estimation implements Parcelable {

        public static Builder builder() {
            return new AutoParcel_ReviewsResponse_Estimation.Builder();
        }

        /**
         * UUID of the MtgObject (content).
         */
        @Nullable
        public abstract String uuid();

        /**
         * Average of all ratings [0..10] (across all languages), Museums ratings includes their
         * Collections ratings.
         */
        @Nullable
        public abstract Float ratingAverage();

        /**
         * Total number of ratings at average snapshot time (across all languages).
         */
        @Nullable
        public abstract Integer ratingsCount();

        /**
         * Total number of reviews at average snapshot time (across all languages).
         */
        @Nullable
        public abstract Integer reviewsCount();

        /**
         * UTC time when average was calculated. Format is according to ISO-8601
         * "YYYY-MM-DDThh:mm:ssZ".
         */
        @Nullable
        public abstract Date date();

        @AutoParcel.Builder
        @JsonPOJOBuilder(withPrefix = "")
        public abstract static class Builder {
            public abstract Builder uuid(String uuid);
            public abstract Builder ratingAverage(Float ratingAverage);
            public abstract Builder ratingsCount(Integer ratingsCount);
            public abstract Builder reviewsCount(Integer reviewsCount);
            public abstract Builder date(Date date);

            public abstract Estimation build();
        }
    }

    /**
     * Pagination to navigate through results data using Unix timestamps.
     */
    @AutoParcel
    public abstract static class Paging implements Parcelable {

        @JsonCreator
        public static Paging create(@JsonProperty("limit") Integer limit,
                                    @JsonProperty("returned_count") Integer returnedCount,
                                    @JsonProperty("total_count") Integer totalCount,
                                    @JsonProperty("next") String next,
                                    @JsonProperty("previous") String previous) {
            return new AutoParcel_ReviewsResponse_Paging(
                    limit, returnedCount, totalCount, next, previous);
        }

        /**
         * The number of individual records that are returned in each page.
         */
        @NonNull
        public abstract Integer limit();

        /**
         * The number of returned records in section 'data'. Values: [0..limit].
         */
        @NonNull
        public abstract Integer returnedCount();

        /**
         * The total number of review records for the content at request time
         * (all languages or for certain language passed as parameter).
         */
        @NonNull
        public abstract Integer totalCount();

        /**
         * An endpoint request that will return the next page of data or null.
         */
        @Nullable
        public abstract String next();

        /**
         * An endpoint request that will return the previous page of data or null.
         */
        @Nullable
        public abstract String previous();
    }
}