/*
 * Copyright (C) 2014 IZITEQ B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package travel.izi.sdk.util;

import travel.izi.sdk.model.enumeration.MediaType;

/**
 * Media files (images, audio, video, etc.) are stored in media directory and scaled\transcoded to
 * needed formats. A media file access link (URL) depends on type of a media. In most cases
 * developer can dynamically build needed media file URL basing on data response parameters.
 */
@SuppressWarnings("unused")
public class MediaHelper {

    public enum ImageSize {

        /**
         * Small size (120x90).
         */
        Small("120x90"),

        /**
         * Medium size (240x180).
         */
        Medium("240x180"),

        /**
         * Large size (480x360).
         */
        Large("480x360"),

        /**
         * Extra large size (800x600).
         */
        XLarge("800x600");

        private final String value;

        @SuppressWarnings("UnnecessaryEnumModifier")
        private ImageSize(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

    }

    private static final String IMAGE_URL_PATTERN = "%1$s/%2$s/%3$s.jpg";
    private static final String FEATURED_IMAGE_URL_PATTERN = "%1$s/featured/%2$s";
    private static final String IMAGE_WITH_SIZE_URL_PATTERN = "%1$s/%2$s/%3$s_%4$s.jpg";

    private static final String AUDIO_URL_PATTERN = "%1$s/%2$s/%3$s.m4a";
    private static final String VIDEO_URL_PATTERN = "%1$s/%2$s/%3$s.mp4";

    /**
     * Media URL.
     */
    private final String mMediaUrl;

    /**
     * Constructor.
     *
     * @param mediaUrl Media URL.
     */
    public MediaHelper(String mediaUrl) {
        mMediaUrl = mediaUrl;
    }

    /**
     * Build URL for images with type {@link MediaType#Story}.
     *
     * @param cpId    UUID of content provider.
     * @param mediaId UUID of media file.
     * @param size    Image size in "WIDTHxHEIGHT" format. E.g., "800x600".
     * @return Image URL
     */
    public String buildImageUrl(String cpId, String mediaId, ImageSize size) {
        return String.format(IMAGE_WITH_SIZE_URL_PATTERN, mMediaUrl, cpId, mediaId, size);
    }

    /**
     * Build URL for images with type {@link MediaType#BrandLogo}, {@link MediaType#BrandCover} or {@link MediaType#Map}.
     *
     * @param cpId    UUID of content provider.
     * @param mediaId UUID of media file.
     * @return Image URL
     */
    public String buildImageUrl(String cpId, String mediaId) {
        return String.format(IMAGE_URL_PATTERN, mMediaUrl, cpId, mediaId);
    }

    /**
     * Build URL for images with type {@link MediaType#Featured}. Characteristics of image: square,
     * max. dimension is 2048x2048 , max. file size is 5Mb, types: jpeg, png.
     *
     * @param mediaId UUID of media file.
     * @return Image URL
     */
    public String buildFeaturedImageUrl(String mediaId) {
        return String.format(FEATURED_IMAGE_URL_PATTERN, mMediaUrl, mediaId);
    }

    /**
     * @param cpId    UUID of content provider.
     * @param mediaId UUID of media file.
     * @return Audio URL
     */
    public String buildAudioUrl(String cpId, String mediaId) {
        return String.format(AUDIO_URL_PATTERN, mMediaUrl, cpId, mediaId);
    }

    /**
     * @param cpId    UUID of content provider.
     * @param mediaId UUID of media file.
     * @return Video URL
     */
    public String buildVideoUrl(String cpId, String mediaId) {
        return String.format(VIDEO_URL_PATTERN, mMediaUrl, cpId, mediaId);
    }

}