package travel.wink.wise.partner.client;

import travel.wink.wise.partner.client.params.ClientId;
import travel.wink.wise.partner.client.params.Code;
import travel.wink.wise.partner.client.params.Email;
import travel.wink.wise.partner.client.params.GrantTypeAuthorizationCode;
import travel.wink.wise.partner.client.params.GrantTypeRefreshToken;
import travel.wink.wise.partner.client.params.GrantTypeRegistrationCode;
import travel.wink.wise.partner.client.params.Parameter;
import travel.wink.wise.partner.client.params.ProfileId;
import travel.wink.wise.partner.client.params.RedirectUri;
import travel.wink.wise.partner.client.params.RefreshToken;
import travel.wink.wise.partner.client.params.RegistrationCode;
import travel.wink.wise.partner.client.params.TargetAccount;
import travel.wink.wise.partner.credentials.api.WiseUser;
import org.springframework.http.ReactiveHttpOutputMessage;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.BodyInserter;
import org.springframework.web.reactive.function.BodyInserters.MultipartInserter;

import java.util.Arrays;
import java.util.Map;

import static java.util.stream.Collectors.toMap;
import static org.springframework.web.reactive.function.BodyInserters.fromMultipartData;
import static org.springframework.web.reactive.function.BodyInserters.fromObject;
import static org.springframework.web.reactive.function.BodyInserters.fromValue;

/**
 * The type Body requests.
 */
public class BodyRequests {

    /**
     * For refresh token multipart inserter.
     *
     * @param refreshToken the refresh token
     * @return the multipart inserter
     */
    public static MultipartInserter forRefreshToken(final RefreshToken refreshToken) {
        return fromMultipartData(multiMap(new GrantTypeRefreshToken(), refreshToken));
    }

    /**
     * For authorization code multipart inserter.
     *
     * @param clientId    the client id
     * @param redirectUri the redirect uri
     * @param code        the code
     * @return the multipart inserter
     */
    public static MultipartInserter forAuthorizationCode(final String clientId, final String redirectUri, final String code) {
        return fromMultipartData(
                multiMap(new GrantTypeAuthorizationCode(),
                        new ClientId(clientId),
                        new Code(code),
                        new RedirectUri(redirectUri)));
    }

    /**
     * For user credentials multipart inserter.
     *
     * @param clientId the client id
     * @param twUser   the tw user
     * @return the multipart inserter
     */
    public static MultipartInserter forUserCredentials(final String clientId, final WiseUser twUser) {
        return fromMultipartData(
                multiMap(new GrantTypeRegistrationCode(),
                        new Email(twUser.getEmail()),
                        new ClientId(clientId),
                        new RegistrationCode(twUser.getRegistrationCode())));
    }

    /**
     * For quote update body inserter.
     *
     * @param profileId     the profile id
     * @param targetAccount the target account
     * @return the body inserter
     */
    public static BodyInserter<Map<String, String>, ReactiveHttpOutputMessage> forQuoteUpdate(final ProfileId profileId, final TargetAccount targetAccount) {
        return fromValue(map(profileId, targetAccount));
    }

    private static Map<String, String> map(final Parameter... parameters) {
        return Arrays.stream(parameters).collect(toMap(Parameter::key, Parameter::value));
    }

    /**
     * Multi map multi value map.
     *
     * @param parameters the parameters
     * @return the multi value map
     */
    static MultiValueMap<String, String> multiMap(final Parameter... parameters) {
        final var multiMap = new LinkedMultiValueMap<String, String>();
        multiMap.setAll(map(parameters));
        return multiMap;
    }
}
