package travel.wink.wise.partner.config;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.http.codec.json.Jackson2JsonEncoder;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.web.reactive.function.client.ExchangeFilterFunction;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;
import travel.wink.wise.partner.credentials.client.WiseAddressClient;
import travel.wink.wise.partner.credentials.client.WiseCredentialsClient;
import travel.wink.wise.partner.credentials.client.WiseProfileClient;
import travel.wink.wise.partner.credentials.client.impl.WiseAddressClientImpl;
import travel.wink.wise.partner.credentials.client.impl.WiseCredentialsClientImpl;
import travel.wink.wise.partner.credentials.client.impl.WiseProfileClientImpl;
import travel.wink.wise.partner.currency.client.WiseCurrencyClient;
import travel.wink.wise.partner.currency.client.impl.WiseCurrencyClientImpl;
import travel.wink.wise.partner.exceptions.WiseDataException;
import travel.wink.wise.partner.exceptions.WiseException;
import travel.wink.wise.partner.exceptions.WiseFaultException;
import travel.wink.wise.partner.quote.client.WiseQuoteClient;
import travel.wink.wise.partner.quote.client.impl.WiseQuoteClientImpl;
import travel.wink.wise.partner.recipient.client.WiseRecipientClient;
import travel.wink.wise.partner.recipient.client.impl.WiseRecipientClientImpl;
import travel.wink.wise.partner.transfer.client.WiseTransferClient;
import travel.wink.wise.partner.transfer.client.impl.WiseTransferClientImpl;

/**
 * The type Wise config.
 */
@Slf4j
@RequiredArgsConstructor
@Configuration
public class WiseConfig {
    private static final ObjectMapper MAPPER = Jackson2ObjectMapperBuilder.json()
            .modules(new JavaTimeModule())
            .build()
            ;
    private final WiseConfigProperties props;

    /**
     * Wise address client wise address client.
     *
     * @return the wise address client
     */
    @Bean("wiseAddressClient")
    public WiseAddressClient wiseAddressClient() {
        return new WiseAddressClientImpl(this.webClient());
    }

    /**
     * Wise credentials client wise credentials client.
     *
     * @return the wise credentials client
     */
    @Bean("wiseCredentialsClient")
    public WiseCredentialsClient wiseCredentialsClient() {
        return new WiseCredentialsClientImpl(this.webClient(), this.props);
    }

    /**
     * Wise profile client wise profile client.
     *
     * @return the wise profile client
     */
    @Bean("wiseProfileClient")
    public WiseProfileClient wiseProfileClient() {
        return new WiseProfileClientImpl(this.webClient());
    }

    /**
     * Wise currency client wise currency client.
     *
     * @return the wise currency client
     */
    @Bean("wiseCurrencyClient")
    public WiseCurrencyClient wiseCurrencyClient() {
        return new WiseCurrencyClientImpl(this.webClient());
    }

    /**
     * Wise quote client wise quote client.
     *
     * @return the wise quote client
     */
    @Bean("wiseQuoteClient")
    public WiseQuoteClient wiseQuoteClient() {
        return new WiseQuoteClientImpl(this.webClient());
    }

    /**
     * Wise recipient client wise recipient client.
     *
     * @return the wise recipient client
     */
    @Bean("wiseRecipientClient")
    public WiseRecipientClient wiseRecipientClient() {
        return new WiseRecipientClientImpl(this.webClient());
    }

    /**
     * Wise transfer client wise transfer client.
     *
     * @return the wise transfer client
     */
    @Bean("wiseTransferClient")
    public WiseTransferClient wiseTransferClient() {
        return new WiseTransferClientImpl(this.webClient());
    }

    /**
     * Web client web client.
     *
     * @return the web client
     */
    @Bean("wiseWebClient")
    public WebClient webClient() {

        final ExchangeStrategies strategies = ExchangeStrategies
                .builder()
                .codecs(clientDefaultCodecsConfigurer -> {
                    clientDefaultCodecsConfigurer.defaultCodecs().jackson2JsonEncoder(new Jackson2JsonEncoder(MAPPER, MediaType.APPLICATION_JSON));
                    clientDefaultCodecsConfigurer.defaultCodecs().jackson2JsonDecoder(new Jackson2JsonDecoder(MAPPER, MediaType.APPLICATION_JSON));
                })
                .build();

        return WebClient.builder()
                .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
                .exchangeStrategies(strategies)
                .filters(exchangeFilterFunctions -> exchangeFilterFunctions.add(logError()))
                .baseUrl(this.props.getBaseUrl())
                .build()
                ;
    }

    /*
    Standard way of handling errors
     */
    private ExchangeFilterFunction logError() {
        return ExchangeFilterFunction.ofResponseProcessor(response -> {
            if (response.statusCode().is4xxClientError() || response.statusCode().is5xxServerError()) {
                return response.bodyToMono(String.class)
                        .flatMap(body -> {
                            log.error("Exception encountered when communicating with Wise SDK: Body is:\n{}", body);
                            try {
                                final WiseDataException data = MAPPER.readValue(body, WiseDataException.class);
                                return Mono.error(WiseException.of(data));
                            } catch (JsonProcessingException e) {
                                return Mono.error(new WiseFaultException(body, response.statusCode().value()));
                            }
                        });
            } else {
                return Mono.just(response);
            }
        });
    }
}
