package travel.wink.wise.partner.credentials.api;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Value;
import org.apache.commons.collections4.CollectionUtils;
import travel.wink.wise.partner.customer.api.Address;

import java.beans.ConstructorProperties;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * The type Wise address.
 */
@Value
public class WiseAddress {
    String firstLine;
    String postCode;
    String city;
    String country;
    String state;
    List<WiseOccupation> occupations;

    /**
     * Instantiates a new Wise address.
     *
     * @param firstLine   the first line
     * @param postCode    the post code
     * @param city        the city
     * @param state       the state
     * @param country     the country
     * @param occupations the occupations
     */
    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    @ConstructorProperties({
            "firstLine",
            "postCode",
            "city",
            "state",
            "country",
            "occupations",
    })
    public WiseAddress(
            @JsonProperty("firstLine") String firstLine,
            @JsonProperty("postCode") String postCode,
            @JsonProperty("city") String city,
            @JsonProperty("state") String state,
            @JsonProperty("country") String country,
            @JsonProperty("occupations") List<WiseOccupation> occupations
    ) {
        this.firstLine = firstLine;
        this.postCode = postCode;
        this.city = city;
        this.state = state;
        this.country = country;
        this.occupations = Objects.requireNonNullElse(occupations, Collections.emptyList());
    }

    /**
     * Of wise address.
     *
     * @param address the address
     * @return the wise address
     */
    public static WiseAddress of(Address address) {
        final List<WiseOccupation> occupations = CollectionUtils.isNotEmpty(address.getOccupations())
                ?
                address.getOccupations()
                        .stream()
                        .map(WiseOccupation::of)
                        .toList()
                :
                Collections.emptyList();
        ;

        return new WiseAddress(
                address.getFirstLine(),
                address.getPostCode(),
                address.getCity(),
                address.getState(),
                address.getCountry(),
                occupations
        );
    }
}
