package travel.wink.wise.partner.credentials.api;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Value;

import java.beans.ConstructorProperties;
import java.time.LocalDateTime;
import java.util.Objects;

import static java.time.LocalDateTime.now;

/**
 * The type Wise profile.
 */
@Value
public class WiseProfile {

    Long twProfileId;
    Long customerId;
    String type;
    ProfileDetails profileDetails;
    LocalDateTime updatedAt;

    /**
     * Instantiates a new Wise profile.
     *
     * @param twProfileId    the tw profile id
     * @param customerId     the customer id
     * @param type           the type
     * @param profileDetails the profile details
     * @param updatedAt      the updated at
     */
    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    @ConstructorProperties({
            "twProfileId",
            "customerId",
            "type",
            "profileDetails",
            "updatedAt"
    })
    public WiseProfile(
            @JsonProperty("twProfileId") Long twProfileId,
            @JsonProperty("customerId") Long customerId,
            @JsonProperty("type") String type,
            @JsonProperty("profileDetails") ProfileDetails profileDetails,
            @JsonProperty("updatedAt") LocalDateTime updatedAt
    ) {
        this.twProfileId = twProfileId;
        this.customerId = customerId;
        this.type = type;
        this.profileDetails = profileDetails;
        this.updatedAt = updatedAt;
    }

    /**
     * Of wise profile.
     *
     * @param twProfileId    the tw profile id
     * @param customerId     the customer id
     * @param type           the type
     * @param profileDetails the profile details
     * @return the wise profile
     */
    public static WiseProfile of(
            Long twProfileId,
            Long customerId,
            String type,
            ProfileDetails profileDetails
    ) {
        return new WiseProfile(
                twProfileId,
                customerId,
                type,
                profileDetails,
                null
        );
    }

    /**
     * Of wise profile.
     *
     * @param twProfileId the tw profile id
     * @param customerId  the customer id
     * @param type        the type
     * @param updatedAt   the updated at
     * @return the wise profile
     */
    public static WiseProfile of(
            Long twProfileId,
            Long customerId,
            String type,
            LocalDateTime updatedAt
    ) {
        return new WiseProfile(
                twProfileId,
                customerId,
                type,
                null,
                updatedAt
        );
    }

    /**
     * Of wise profile.
     *
     * @param customerId the customer id
     * @param response   the response
     * @return the wise profile
     */
    public static WiseProfile of(Long customerId, WiseProfileResponse response) {
        return new WiseProfile(
                response.getId(),
                customerId,
                response.getType(),
                response.getDetails() != null ? ProfileDetails.of(response.getDetails()) : null,
                null
        );
    }

    /**
     * Update updated at wise profile.
     *
     * @param updatedAt the updated at
     * @return the wise profile
     */
    public WiseProfile updateUpdatedAt(final LocalDateTime updatedAt) {
        return new WiseProfile(
                this.twProfileId,
                this.customerId,
                this.type,
                this.profileDetails,
                updatedAt
        );
    }

    /**
     * Needs yearly update boolean.
     *
     * @return the boolean
     */
    public boolean needsYearlyUpdate() {
        return updatedAt == null || updatedAt.isBefore(now().minusYears(1L));
    }
}
