package travel.wink.wise.partner.credentials.client.impl;

import lombok.RequiredArgsConstructor;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;
import travel.wink.wise.partner.config.WiseConfigProperties;
import travel.wink.wise.partner.client.params.RefreshToken;
import travel.wink.wise.partner.credentials.api.WiseClientCredentials;
import travel.wink.wise.partner.credentials.api.WiseSignUpRequest;
import travel.wink.wise.partner.credentials.api.WiseUserResponse;
import travel.wink.wise.partner.credentials.api.WiseUserTokensResponse;
import travel.wink.wise.partner.credentials.client.WiseCredentialsClient;
import travel.wink.wise.partner.credentials.api.WiseUser;
import travel.wink.wise.partner.credentials.api.WiseUserTokens;

import java.util.Base64;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.time.ZonedDateTime.now;
import static org.springframework.http.HttpHeaders.AUTHORIZATION;
import static org.springframework.http.MediaType.APPLICATION_FORM_URLENCODED;
import static org.springframework.http.MediaType.APPLICATION_JSON;
import static travel.wink.wise.partner.client.BodyRequests.forAuthorizationCode;
import static travel.wink.wise.partner.client.BodyRequests.forRefreshToken;
import static travel.wink.wise.partner.client.BodyRequests.forUserCredentials;
import static travel.wink.wise.partner.client.TransferWisePaths.OAUTH_TOKEN_PATH;
import static travel.wink.wise.partner.client.TransferWisePaths.SIGNUP_PATH;

/**
 * The type Wise credentials client.
 */
@RequiredArgsConstructor
public class WiseCredentialsClientImpl implements WiseCredentialsClient {

    private static final String GRANT_TYPE_CLIENT_CREDENTIALS = "grant_type=client_credentials";
//
//    @Value("${twbank.clientId}")
//    private String clientId;
//
//    @Value("${twbank.secret}")
//    private String secret;
//
//    @Value("${twbank.redirectUri}")
//    private String redirectUri;

    private final WebClient client;
    private final WiseConfigProperties props;

    @Override
    public Mono<WiseUser> signUp(
            final String email,
            final String registrationCode
    ) {

        return twClientCredentials()
                .flatMap(twClientCredentials -> signUp(email, registrationCode, twClientCredentials));
    }

    @Override
    public Mono<WiseUserTokens> getUserTokensForCode(
            final String code,
            final Long customerId
    ) {
        return this.client.post()
                .uri(OAUTH_TOKEN_PATH)
                .header(AUTHORIZATION, basicAuth())
                .body(forAuthorizationCode(this.props.getClientId(), this.props.getRedirectUri(), code))
                .retrieve()
                .bodyToMono(WiseUserTokensResponse.class)
                .map(twUserTokensResponse -> new WiseUserTokens(customerId,
                        null,
                        twUserTokensResponse.getAccessToken(),
                        twUserTokensResponse.getRefreshToken(),
                        now().plusSeconds(twUserTokensResponse.getExpiresIn())));
    }

    @Override
    public Mono<WiseUserTokens> getUserTokens(final WiseUser twUser) {
        return this.client.post()
                .uri(OAUTH_TOKEN_PATH)
                .contentType(APPLICATION_FORM_URLENCODED)
                .header(AUTHORIZATION, basicAuth())
                .body(forUserCredentials(this.props.getClientId(), twUser))
                .retrieve()
                .bodyToMono(WiseUserTokensResponse.class)
                .map(twUserTokensResponse -> new WiseUserTokens(twUser.getCustomerId(),
                        twUser.getTwUserId(),
                        twUserTokensResponse.getAccessToken(),
                        twUserTokensResponse.getRefreshToken(),
                        now().plusSeconds(twUserTokensResponse.getExpiresIn())));
    }

    @Override
    public Mono<WiseUserTokens> refresh(final WiseUserTokens twUserTokens) {
        return this.client.post()
                .uri(OAUTH_TOKEN_PATH)
                .header(AUTHORIZATION, basicAuth())
                .body(forRefreshToken(new RefreshToken(twUserTokens.getRefreshToken())))
                .retrieve()
                .bodyToMono(WiseUserTokensResponse.class)
                .map(twUserTokensResponse -> new WiseUserTokens(twUserTokens.getCustomerId(),
                        twUserTokens.getTwUserId(),
                        twUserTokensResponse.getAccessToken(),
                        twUserTokensResponse.getRefreshToken(),
                        now().plusSeconds(twUserTokensResponse.getExpiresIn())));
    }

    private Mono<WiseClientCredentials> twClientCredentials() {
        return this.client.post()
                .uri(OAUTH_TOKEN_PATH)
                .contentType(APPLICATION_FORM_URLENCODED)
                .header(AUTHORIZATION, basicAuth())
                .bodyValue(GRANT_TYPE_CLIENT_CREDENTIALS)
                .retrieve()
                .bodyToMono(WiseClientCredentials.class);
    }

    private Mono<WiseUser> signUp(
            String email,
            String registrationCode,
            WiseClientCredentials twClientCredentials
    ) {
        return this.client.post()
                .uri(SIGNUP_PATH)
                .contentType(APPLICATION_JSON)
                .header(AUTHORIZATION, twClientCredentials.bearer())
                .bodyValue(new WiseSignUpRequest(email, registrationCode))
                .retrieve()
                .bodyToMono(WiseUserResponse.class)
                .map(twUserResponse -> WiseUser.of(twUserResponse.getId(),
                        twUserResponse.getEmail(),
                        twUserResponse.getActive()));
    }


    private String basicAuth() {
        return String.format("Basic %s", encodedCredentials());
    }

    private String encodedCredentials() {
        return Base64.getEncoder().encodeToString(credentials().getBytes(UTF_8));
    }

    private String credentials() {
        return this.props.getClientId() + ":" + this.props.getClientSecret();
    }

}
