package travel.wink.wise.partner.credentials.client.impl;

import lombok.RequiredArgsConstructor;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import travel.wink.wise.partner.credentials.api.CreatePersonalProfile;
import travel.wink.wise.partner.credentials.api.ProfileDetails;
import travel.wink.wise.partner.credentials.api.UpdatePersonalProfile;
import travel.wink.wise.partner.credentials.api.WiseProfile;
import travel.wink.wise.partner.credentials.api.WiseProfileDetails;
import travel.wink.wise.partner.credentials.api.WiseProfileResponse;
import travel.wink.wise.partner.credentials.api.WiseUserTokens;
import travel.wink.wise.partner.credentials.client.WiseProfileClient;

import static org.springframework.http.HttpHeaders.AUTHORIZATION;
import static org.springframework.http.MediaType.APPLICATION_JSON;
import static travel.wink.wise.partner.client.TransferWisePaths.PROFILES_PATH_V1;
import static travel.wink.wise.partner.client.TransferWisePaths.getProfilePath;
import static travel.wink.wise.partner.client.TransferWisePaths.updateWindowPath;

/**
 * The type Wise profile client.
 */
@RequiredArgsConstructor
public class WiseProfileClientImpl implements WiseProfileClient {

    private final WebClient client;

    @Override
    public Mono<WiseProfile> createPersonalProfile(
            WiseUserTokens twUserTokens,
            CreatePersonalProfile createPersonalProfile
    ) {
        return this.client.post()
                .uri(PROFILES_PATH_V1)
                .contentType(APPLICATION_JSON)
                .header(AUTHORIZATION, twUserTokens.bearer())
                .bodyValue(createPersonalProfile)
                .retrieve()
                .bodyToMono(WiseProfileResponse.class)
                .map(response -> WiseProfile.of(twUserTokens.getCustomerId(), response))
                ;
    }

    @Override
    public Mono<WiseProfile> updatePersonalProfile(
            WiseUserTokens twUserTokens,
            UpdatePersonalProfile updatePersonalProfile
    ) {
        return this.client.put()
                .uri(PROFILES_PATH_V1)
                .contentType(APPLICATION_JSON)
                .header(AUTHORIZATION, twUserTokens.bearer())
                .bodyValue(updatePersonalProfile)
                .retrieve()
                .bodyToMono(WiseProfileResponse.class)
                .map(response -> WiseProfile.of(twUserTokens.getCustomerId(), response))
                ;
    }

    @Override
    public Mono<WiseProfile> getProfile(
            final WiseUserTokens twUserTokens,
            final Long twProfileId
    ) {
        return this.client.get()
                .uri(getProfilePath(twProfileId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToMono(WiseProfileResponse.class)
                .map(response -> WiseProfile.of(twUserTokens.getCustomerId(), response))
                ;
    }

    @Override
    public Flux<WiseProfile> getProfiles(final WiseUserTokens twUserTokens) {
        return this.client.get()
                .uri(PROFILES_PATH_V1)
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToFlux(WiseProfileResponse.class)
                .map(response -> WiseProfile.of(twUserTokens.getCustomerId(), response))
                ;
    }

    @Override
    public Mono<Boolean> openUpdateWindow(
            WiseUserTokens twUserTokens,
            Long twProfileId
    ) {
        return this.client.post()
                .uri(updateWindowPath(twProfileId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .exchangeToMono(clientResponse -> Mono.just(clientResponse.statusCode().is2xxSuccessful()))
                ;
    }

    @Override
    public Mono<Boolean> closeUpdateWindow(
            WiseUserTokens twUserTokens,
            Long twProfileId
    ) {
        return this.client.delete()
                .uri(updateWindowPath(twProfileId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .exchangeToMono(clientResponse -> Mono.just(clientResponse.statusCode().is2xxSuccessful()))
                ;
    }

}
