package travel.wink.wise.partner.quote.api;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Value;

import java.beans.ConstructorProperties;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * The type Payment option.
 */
@Value
public class PaymentOption {
    /**
     * The constant BANK_TRANSFER.
     */
    public static final String BANK_TRANSFER = "BANK_TRANSFER";
    /**
     * The constant SWIFT.
     */
    public static final String SWIFT = "SWIFT";

    Boolean disabled;

    LocalDateTime estimatedDelivery;
    String formattedEstimatedDelivery;

    List<String> estimatedDeliveryDelays;
    Fee fee;

    BigDecimal sourceAmount;
    BigDecimal targetAmount;
    String payIn;
    String payOut;

    List<String> allowedProfileTypes;
    DisabledReason disabledReason;

    /**
     * Instantiates a new Payment option.
     *
     * @param disabled                   the disabled
     * @param estimatedDelivery          the estimated delivery
     * @param formattedEstimatedDelivery the formatted estimated delivery
     * @param estimatedDeliveryDelays    the estimated delivery delays
     * @param fee                        the fee
     * @param sourceAmount               the source amount
     * @param targetAmount               the target amount
     * @param payIn                      the pay in
     * @param payOut                     the pay out
     * @param allowedProfileTypes        the allowed profile types
     * @param disabledReason             the disabled reason
     */
    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    @ConstructorProperties({
            "disabled",
            "estimatedDelivery",
            "formattedEstimatedDelivery",
            "estimatedDeliveryDelays",
            "fee",
            "sourceAmount",
            "targetAmount",
            "payIn",
            "payOut",
            "allowedProfileTypes",
            "disabledReason"
    })
    public PaymentOption(
            @JsonProperty("disabled") Boolean disabled,
            @JsonProperty("estimatedDelivery") @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss'Z'") LocalDateTime estimatedDelivery,
            @JsonProperty("formattedEstimatedDelivery") String formattedEstimatedDelivery,
            @JsonProperty("estimatedDeliveryDelays") List<String> estimatedDeliveryDelays,
            @JsonProperty("fee") Fee fee,
            @JsonProperty("sourceAmount") BigDecimal sourceAmount,
            @JsonProperty("targetAmount") BigDecimal targetAmount,
            @JsonProperty("payIn") String payIn,
            @JsonProperty("payOut") String payOut,
            @JsonProperty("allowedProfileTypes") List<String> allowedProfileTypes,
            @JsonProperty("disabledReason") DisabledReason disabledReason
    ) {
        this.disabled = disabled;
        this.estimatedDelivery = estimatedDelivery;
        this.formattedEstimatedDelivery = formattedEstimatedDelivery;
        this.estimatedDeliveryDelays = Objects.requireNonNullElse(estimatedDeliveryDelays, Collections.emptyList());
        this.fee = fee;
        this.sourceAmount = sourceAmount;
        this.targetAmount = targetAmount;
        this.payIn = payIn;
        this.payOut = payOut;
        this.allowedProfileTypes = Objects.requireNonNullElse(allowedProfileTypes, Collections.emptyList());
        this.disabledReason = disabledReason;
    }

    /**
     * Of payment option.
     *
     * @param payIn  the pay in
     * @param payOut the pay out
     * @return the payment option
     */
    public static PaymentOption of(final String payIn, final String payOut) {
        return new PaymentOption(
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                payIn,
                payOut,
                null,
                null
        );
    }

    /**
     * Is swift boolean.
     *
     * @return the boolean
     */
    public boolean isSwift() {
        return payIn.equals(SWIFT) && payOut.equals(BANK_TRANSFER);
    }

    /**
     * Is bank transfer boolean.
     *
     * @return the boolean
     */
    public boolean isBankTransfer() {
        return payIn.equals(BANK_TRANSFER) && payOut.equals(BANK_TRANSFER);
    }
}
