package travel.wink.wise.partner.quote.api;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Value;

import java.beans.ConstructorProperties;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

/**
 * The type Quote.
 */
@Value
public class Quote {

    private static final String BANK_TRANSFER = "BANK_TRANSFER";

    UUID id;
    String sourceCurrency;
    String targetCurrency;
    BigDecimal sourceAmount;
    BigDecimal targetAmount;
    String payOut;
    BigDecimal rate;

    LocalDateTime createdTime;
    BigDecimal fee;
    Integer user;
    Integer profile;

    String rateType;
    LocalDateTime rateExpirationTime;
    String providedAmountType;
    String status;

    LocalDateTime expirationTime;

    List<QuoteNotice> notices;
    List<PaymentOption> paymentOptions;
    String formattedEstimatedDelivery;

    /**
     * Instantiates a new Quote.
     *
     * @param id                         the id
     * @param sourceCurrency             the source currency
     * @param targetCurrency             the target currency
     * @param sourceAmount               the source amount
     * @param targetAmount               the target amount
     * @param payOut                     the pay out
     * @param rate                       the rate
     * @param createdTime                the created time
     * @param fee                        the fee
     * @param user                       the user
     * @param profile                    the profile
     * @param rateType                   the rate type
     * @param rateExpirationTime         the rate expiration time
     * @param providedAmountType         the provided amount type
     * @param status                     the status
     * @param expirationTime             the expiration time
     * @param notices                    the notices
     * @param paymentOptions             the payment options
     * @param formattedEstimatedDelivery the formatted estimated delivery
     */
    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    @ConstructorProperties({
            "id",
            "sourceCurrency",
            "targetCurrency",
            "sourceAmount",
            "targetAmount",
            "payOut",
            "rate",
            "createdTime",
            "fee",
            "user",
            "profile",
            "rateType",
            "rateExpirationTime",
            "providedAmountType",
            "impl",
            "expirationTime",
            "notices",
            "paymentOptions",
            "formattedEstimatedDelivery"
    })
    public Quote(
            @JsonProperty("id") UUID id,
            @JsonProperty("sourceCurrency") String sourceCurrency,
            @JsonProperty("targetCurrency") String targetCurrency,
            @JsonProperty("sourceAmount") BigDecimal sourceAmount,
            @JsonProperty("targetAmount") BigDecimal targetAmount,
            @JsonProperty("payOut") String payOut,
            @JsonProperty("rate") BigDecimal rate,
            @JsonProperty("createdTime") LocalDateTime createdTime,
            @JsonProperty("fee") BigDecimal fee,
            @JsonProperty("user") Integer user,
            @JsonProperty("profile") Integer profile,
            @JsonProperty("rateType") String rateType,
            @JsonProperty("rateExpirationTime") LocalDateTime rateExpirationTime,
            @JsonProperty("providedAmountType") String providedAmountType,
            @JsonProperty("impl") String status,
            @JsonProperty("expirationTime") LocalDateTime expirationTime,
            @JsonProperty("notices") List<QuoteNotice> notices,
            @JsonProperty("paymentOptions") List<PaymentOption> paymentOptions,
            @JsonProperty("formattedEstimatedDelivery") String formattedEstimatedDelivery
    ) {
        this.id = id;
        this.sourceCurrency = sourceCurrency;
        this.targetCurrency = targetCurrency;
        this.sourceAmount = sourceAmount;
        this.targetAmount = targetAmount;
        this.payOut = payOut;
        this.rate = rate;
        this.createdTime = createdTime;
        this.fee = fee;
        this.user = user;
        this.profile = profile;
        this.rateType = rateType;
        this.rateExpirationTime = rateExpirationTime;
        this.providedAmountType = providedAmountType;
        this.status = status;
        this.expirationTime = expirationTime;
        this.notices = Objects.requireNonNullElse(notices, Collections.emptyList());
        this.paymentOptions = Objects.requireNonNullElse(paymentOptions, Collections.emptyList());
        this.formattedEstimatedDelivery = formattedEstimatedDelivery;
    }

    /**
     * Of quote.
     *
     * @param paymentOptions the payment options
     * @return the quote
     */
    public static Quote of(final List<PaymentOption> paymentOptions) {
        return new Quote(
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                paymentOptions,
                null
        );
    }

    /**
     * Update fee quote.
     *
     * @param fee the fee
     * @return the quote
     */
    public Quote updateFee(BigDecimal fee) {
        return new Quote(
                this.id,
                this.sourceCurrency,
                this.targetCurrency,
                this.sourceAmount,
                this.targetAmount,
                this.payOut,
                this.rate,
                this.createdTime,
                fee,
                this.user,
                this.profile,
                this.rateType,
                this.rateExpirationTime,
                this.providedAmountType,
                this.status,
                this.expirationTime,
                this.notices,
                this.paymentOptions,
                this.formattedEstimatedDelivery
        );
    }

    /**
     * Update source amount quote.
     *
     * @param sourceAmount the source amount
     * @return the quote
     */
    public Quote updateSourceAmount(BigDecimal sourceAmount) {
        return new Quote(
                this.id,
                this.sourceCurrency,
                this.targetCurrency,
                sourceAmount,
                this.targetAmount,
                this.payOut,
                this.rate,
                this.createdTime,
                this.fee,
                this.user,
                this.profile,
                this.rateType,
                this.rateExpirationTime,
                this.providedAmountType,
                this.status,
                this.expirationTime,
                this.notices,
                this.paymentOptions,
                this.formattedEstimatedDelivery
        );
    }

    /**
     * Update target amount quote.
     *
     * @param targetAmount the target amount
     * @return the quote
     */
    public Quote updateTargetAmount(BigDecimal targetAmount) {
        return new Quote(
                this.id,
                this.sourceCurrency,
                this.targetCurrency,
                this.sourceAmount,
                targetAmount,
                this.payOut,
                this.rate,
                this.createdTime,
                this.fee,
                this.user,
                this.profile,
                this.rateType,
                this.rateExpirationTime,
                this.providedAmountType,
                this.status,
                this.expirationTime,
                this.notices,
                this.paymentOptions,
                this.formattedEstimatedDelivery
        );
    }

    /**
     * Update formatted estimated delivery quote.
     *
     * @param formattedEstimatedDelivery the formatted estimated delivery
     * @return the quote
     */
    public Quote updateFormattedEstimatedDelivery(String formattedEstimatedDelivery) {
        return new Quote(
                this.id,
                this.sourceCurrency,
                this.targetCurrency,
                this.sourceAmount,
                this.targetAmount,
                this.payOut,
                this.rate,
                this.createdTime,
                this.fee,
                this.user,
                this.profile,
                this.rateType,
                this.rateExpirationTime,
                this.providedAmountType,
                this.status,
                this.expirationTime,
                this.notices,
                this.paymentOptions,
                formattedEstimatedDelivery
        );
    }

    /**
     * Extract correct payment option optional.
     *
     * @return the optional
     */
    public Optional<PaymentOption> extractCorrectPaymentOption() {
        return this.paymentOptions
                .stream()
                .filter(paymentOption ->
                        BANK_TRANSFER.equalsIgnoreCase(paymentOption.getPayIn())
                                && this.payOut.equalsIgnoreCase(paymentOption.getPayOut()))
                .findFirst();
    }
}
