package travel.wink.wise.partner.quote.client.impl;

import lombok.RequiredArgsConstructor;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;
import travel.wink.wise.partner.client.params.ProfileId;
import travel.wink.wise.partner.client.params.TargetAccount;
import travel.wink.wise.partner.credentials.api.WiseUserTokens;
import travel.wink.wise.partner.quote.api.WiseQuoteResponse;
import travel.wink.wise.partner.quote.api.CreateAnonymousQuote;
import travel.wink.wise.partner.quote.api.CreateQuote;
import travel.wink.wise.partner.quote.api.Quote;
import travel.wink.wise.partner.quote.client.WiseQuoteClient;

import java.util.UUID;
import java.util.function.Function;

import static org.springframework.http.HttpHeaders.AUTHORIZATION;
import static org.springframework.http.MediaType.APPLICATION_JSON;
import static travel.wink.wise.partner.client.BodyRequests.forQuoteUpdate;
import static travel.wink.wise.partner.client.TransferWisePaths.QUOTES_PATH_V2;
import static travel.wink.wise.partner.client.TransferWisePaths.quotesPathV2;

/**
 * The type Wise quote client.
 */
@RequiredArgsConstructor
public class WiseQuoteClientImpl implements WiseQuoteClient {

    private static final MediaType MERGE_PATCH_JSON = MediaType.valueOf("application/merge-patch+json");
    private final WebClient client;

    @Override
    public Mono<Quote> createAnonymousQuote(final CreateAnonymousQuote createAnonymousQuote) {
        return this.client.post()
                .uri(QUOTES_PATH_V2)
                .contentType(APPLICATION_JSON)
                .bodyValue(createAnonymousQuote)
                .retrieve()
                .bodyToMono(WiseQuoteResponse.class)
                .map(twResponseToQuote);
    }

    @Override
    public Mono<Quote> createQuote(
            final WiseUserTokens twUserTokens,
            final CreateQuote createQuote
    ) {
        return this.client.post()
                .uri(QUOTES_PATH_V2)
                .header(AUTHORIZATION, twUserTokens.bearer())
                .contentType(APPLICATION_JSON)
                .bodyValue(createQuote)
                .retrieve()
                .bodyToMono(WiseQuoteResponse.class)
                .map(twResponseToQuote);
    }

    @Override
    public Mono<Quote> updateQuote(
            final WiseUserTokens twUserTokens,
            final UUID quoteId,
            final Long profileId,
            final Long recipientId
    ) {
        return this.client.patch()
                .uri(quotesPathV2(quoteId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .contentType(MERGE_PATCH_JSON)
                .body(forQuoteUpdate(new ProfileId(profileId), new TargetAccount(recipientId)))
                .retrieve()
                .bodyToMono(WiseQuoteResponse.class)
                .map(twResponseToQuote);
    }

    @Override
    public Mono<Quote> getQuote(
            WiseUserTokens twUserTokens,
            UUID quoteId
    ) {
        return this.client.get()
                .uri(quotesPathV2(quoteId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToMono(WiseQuoteResponse.class)
                .map(twResponseToQuote);
    }

    private final Function<WiseQuoteResponse, Quote> twResponseToQuote =
            twQuoteResponse ->
                    new Quote(twQuoteResponse.getId(),
                            twQuoteResponse.getSourceCurrency(),
                            twQuoteResponse.getTargetCurrency(),
                            twQuoteResponse.getSourceAmount(),
                            twQuoteResponse.getTargetAmount(),
                            twQuoteResponse.getPayOut(),
                            twQuoteResponse.getRate(),
                            twQuoteResponse.getCreatedTime(),
                            twQuoteResponse.getFee(),
                            twQuoteResponse.getUser(),
                            twQuoteResponse.getProfile(),
                            twQuoteResponse.getRateType(),
                            twQuoteResponse.getRateExpirationTime(),
                            twQuoteResponse.getProvidedAmountType(),
                            twQuoteResponse.getStatus(),
                            twQuoteResponse.getExpirationTime(),
                            twQuoteResponse.getNotices(),
                            twQuoteResponse.getPaymentOptions(),
                            null
                    );
}
