package travel.wink.wise.partner.recipient.client.impl;

import lombok.RequiredArgsConstructor;
import travel.wink.wise.partner.credentials.api.WiseUserTokens;
import travel.wink.wise.partner.recipient.api.WiseRecipientsContent;
import travel.wink.wise.partner.recipient.api.Recipient;
import travel.wink.wise.partner.recipient.client.WiseRecipientClient;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.UUID;

import static org.springframework.http.MediaType.APPLICATION_JSON;
import static travel.wink.wise.partner.client.TransferWisePaths.RECIPIENTS_PATH_V1;
import static travel.wink.wise.partner.client.TransferWisePaths.RECIPIENTS_PATH_V2;
import static travel.wink.wise.partner.client.TransferWisePaths.recipientByIdPath;
import static travel.wink.wise.partner.client.TransferWisePaths.recipientRequirementsPath;
import static org.springframework.http.HttpHeaders.AUTHORIZATION;
import static reactor.core.publisher.Flux.fromIterable;

/**
 * The type Wise recipient client.
 */
@RequiredArgsConstructor
public class WiseRecipientClientImpl implements WiseRecipientClient {

    private final WebClient client;

    @Override
    public Mono<Recipient> getRecipient(final WiseUserTokens twUserTokens, final Long recipientId) {
        return this.client.get()
                .uri(recipientByIdPath(recipientId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToMono(Recipient.class);
    }

    @Override
    public Flux<Recipient> getAllRecipients(final WiseUserTokens twUserTokens, final Long twProfileId, final String currencyCode) {
        return this.client.get()
                .uri(uriBuilder -> {
                    uriBuilder = uriBuilder
                            .path(RECIPIENTS_PATH_V2)
                            .queryParam("profile", twProfileId);

                    if (currencyCode != null) {
                        uriBuilder = uriBuilder.queryParam("currency", currencyCode);
                    }

                    return uriBuilder.build();

                })
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToFlux(WiseRecipientsContent.class)
                .flatMap(content -> fromIterable(content.getRecipients()));

    }

    @Override
    public Mono<String> create(final WiseUserTokens twUserTokens, final String requestBody) {
        return this.client.post()
                .uri(RECIPIENTS_PATH_V1)
                .header(AUTHORIZATION, twUserTokens.bearer())
                .contentType(APPLICATION_JSON)
                .bodyValue(requestBody)
                .retrieve()
                .bodyToMono(String.class);
    }

    @Override
    public Mono<String> getRequirements(WiseUserTokens twUserTokens, UUID quoteId) {
        return this.client.get()
                .uri(recipientRequirementsPath(quoteId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .retrieve()
                .bodyToMono(String.class);
    }

    @Override
    public Mono<String> createRequirements(WiseUserTokens twUserTokens, UUID quoteId, String requestBody) {
        return this.client.post()
                .uri(recipientRequirementsPath(quoteId))
                .header(AUTHORIZATION, twUserTokens.bearer())
                .contentType(APPLICATION_JSON)
                .bodyValue(requestBody)
                .retrieve()
                .bodyToMono(String.class);
    }

}
