/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.operation

import kotlin.time.Duration
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

@OptIn(ExperimentalTime::class)
public val normalizedEmissionEntryComparator: Comparator<NormalizedEmissionEntry> =
    compareBy<NormalizedEmissionEntry> { it.ratingPerMinute }.thenBy { it.time }

public interface NormalizedEmissionEntry : Comparable<NormalizedEmissionEntry> {
    @OptIn(ExperimentalTime::class)
    public val time: Instant
    public val ratingPerMinute: Double
    public val duration: Duration

    public val ratingSummed: Double
    public val ratingCalulated: Double

    @OptIn(ExperimentalTime::class)
    public override operator fun compareTo(other: NormalizedEmissionEntry): Int {
        return normalizedEmissionEntryComparator.compare(this, other)
    }
}

public interface NormalizedEmissions : Iterable<NormalizedEmissionEntry> {
    public val entries: List<NormalizedEmissionEntry>
}

/**
 * Aggregates and normalizes a list of `EmissionEntry` instances into a
 * set of `NormalizedEmissionEntry` objects based on the specified target
 * duration.
 *
 * Each normalized entry represents a time-windowed aggregation of
 * emissions whose combined duration meets or exceeds the given target
 * duration.
 *
 * @param targetDuration the duration threshold for aggregating emissions
 *    into individual normalized entries.
 * @return a `NormalizedEmissions` instance containing the resulting set of
 *    aggregated emissions.
 * @throws IllegalArgumentException if the emission data is not a
 *    continuous, chronological time-series
 */
@OptIn(ExperimentalTime::class)
public fun List<EmissionEntry>.normalize(
    targetDuration: Duration,
): NormalizedEmissions {
    val windows = mutableListOf<WindowedEmissionEntry>()
    val sorted = sortedBy { it.time }

    var last: EmissionEntry? = null
    for (current in sorted) {
        last = last.requireConsecutiveTime(current)
        windows.forEach { it.add(current) }
        windows.add(WindowedEmissionEntry(emission = current, windowDuration = targetDuration))
    }

    return WindowedEmissions(windows.filter { it.duration >= targetDuration })
}

@OptIn(ExperimentalTime::class)
internal fun EmissionEntry?.requireConsecutiveTime(next: EmissionEntry): EmissionEntry {
    if (this != null) {
        require(endTime == next.time) { "Emissisons must be consecutive in time. Found: $endTime, ${next.time}." }
    }
    return next
}

internal class WindowedEmissions(
    override val entries: List<NormalizedEmissionEntry>,
) : NormalizedEmissions, List<NormalizedEmissionEntry> by entries {
    override fun toString(): String {
        return "WindowedEmissions(entries=$entries)"
    }
}

/**
 * A class representing a time-based window of emission entries. This
 * class aggregates multiple `EmissionEntry` objects into a single entry,
 * ensuring that their combined duration does not exceed the specified
 * window duration.
 *
 * @constructor Initializes the window with an initial emission entry. The
 *    window begins with a single `EmissionEntry` and can be extended using
 *    the `add` method, as long as the combined duration stays within the
 *    window duration limit.
 *
 * This class calculates the average rating and updates the total duration
 * as more entries are added.
 * - The `time` property represents the time of the first emission in the
 *   window.
 * - The `rating` property is the average rating of all emissions in the
 *   window.
 * - The `duration` property is the total duration of all emissions in the
 *   window.
 *
 * @property windowDuration the maximum duration for the time window.
 */
internal class WindowedEmissionEntry(
    emission: EmissionEntry,
    val windowDuration: Duration,
) : NormalizedEmissionEntry {
    private val _emissions = mutableListOf(emission)

    @OptIn(ExperimentalTime::class)
    override val time = emission.time

    override var ratingPerMinute: Double = emission.ratingPerMinute
        private set

    override var duration: Duration = emission.duration
        private set

    override val ratingSummed: Double
        get() = _emissions.sumOf { it.rating }

    override val ratingCalulated: Double
        get() = duration.inWholeMinutes * ratingPerMinute

    fun add(emission: EmissionEntry) {
        if (duration < windowDuration) {
            _emissions.add(emission)
            duration += emission.duration
            ratingPerMinute = _emissions.map { it.ratingPerMinute }.average()
            return
        }
    }

    @OptIn(ExperimentalTime::class)
    override fun toString(): String {
        return "WindowedEmissions(time=$time, ratingPerMinute=$ratingPerMinute, duration=$duration)"
    }
}
