/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon

import arrow.core.getOrElse
import ventures.dvc.karbon.location.ForecastLocation
import ventures.dvc.karbon.operation.normalize
import ventures.dvc.karbon.operation.normalizedEmissionEntryComparator
import ventures.dvc.karbon.operation.toEmissionEntry
import ventures.dvc.karbon.provider.ForecastRequest
import ventures.dvc.karbon.provider.KarbonForecastProvider
import kotlin.time.Duration

public interface KarbonAware {
    public suspend fun estimateBestExecutionTime(
        estimatedTaskDuration: Duration,
        location: ForecastLocation,
    )
}

public fun KarbonAware(provider: KarbonForecastProvider): KarbonAware {
    return CoreKarbonAwareCalulator(provider)
}


private class CoreKarbonAwareCalulator(
    private val provider: KarbonForecastProvider,
) : KarbonAware {
    override suspend fun estimateBestExecutionTime(
        estimatedTaskDuration: Duration,
        location: ForecastLocation,
    ) {
        println("KarbonAware: estimateBestExecutionTime: $estimatedTaskDuration, $location")

        val result = provider(ForecastRequest(location = location))
        val container = result.getOrElse { t ->
            println("KarbonAware: could not get emissions: $t")
            throw RuntimeException("error getting emissions")
        }

        val emissionEntries = container.emissions.map {
            it.toEmissionEntry()
        }
        println("KarbonAware: emissionEntries: $emissionEntries")

        val normalizedEmissions = emissionEntries.normalize(targetDuration = estimatedTaskDuration)
        println("KarbonAware: normalizedEmissions: $normalizedEmissions")

        val firstEntry = normalizedEmissions.sortedWith(normalizedEmissionEntryComparator).firstOrNull()
        println("KarbonAware: firstEntry: $firstEntry")


        println("KarbonAware: done")
    }
}