/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.location

public sealed interface ForecastLocation {
    public val code: String
    public val known: Boolean

    public companion object {
        public fun of(code: String): ForecastLocation =
            KnownForecastLocation.of(code) ?: CustomForecastLocation(code)
    }

}

public class KnownForecastLocation
private constructor(
    override val code: String,
) : ForecastLocation {

    override val known: Boolean = true

    public companion object {
        public val Austria: KnownForecastLocation = KnownForecastLocation("at")
        public val France: KnownForecastLocation = KnownForecastLocation("fr")
        public val Germany: KnownForecastLocation = KnownForecastLocation("de")
        public val Switzerland: KnownForecastLocation = KnownForecastLocation("ch")
        public val UnitedKingdom: KnownForecastLocation = KnownForecastLocation("uk")

        public val all: List<KnownForecastLocation> = listOf(Austria, France, Germany, Switzerland, UnitedKingdom)

        public fun of(code: String): KnownForecastLocation? = all.find { it.code == code }
    }

    override fun toString(): String {
        return "KnownForecastLocation(code='$code')"
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as KnownForecastLocation

        return code == other.code
    }

    override fun hashCode(): Int {
        return code.hashCode()
    }
}


public class CustomForecastLocation(override val code: String) : ForecastLocation {

    override val known: Boolean = false

    override fun toString(): String {
        return "CustomForecastLocation(code='$code')"
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as CustomForecastLocation

        return code == other.code
    }

    override fun hashCode(): Int {
        return code.hashCode()
    }
}

