/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.provider

import arrow.core.Either
import ventures.dvc.karbon.emission.ForecastEmission
import ventures.dvc.karbon.location.ForecastLocation
import kotlin.jvm.JvmInline

public fun interface KarbonForecastProvider {
    public suspend operator fun invoke(request: ForecastRequest): Either<ForecastError, ForecastResponse>
}

public interface ForecastRequest {
    public val location: ForecastLocation
}

public interface ForecastResponse {
    public val key: ForecastKey
    public val emissions: List<ForecastEmission>
}

public fun ForecastResponse(key: ForecastKey, emissions: List<ForecastEmission>): ForecastResponse =
    CoreForecastResponse(key = key, emissions = emissions)

public fun ForecastRequest(location: ForecastLocation): ForecastRequest =
    CoreForecastRequest(
        location = location
    )

private data class CoreForecastResponse(
    override val key: ForecastKey,
    override val emissions: List<ForecastEmission>,
) : ForecastResponse {}


public interface ForecastKey {
    public val location: ForecastLocation
    public val hash: ForecastHash
}

public fun ForecastKey(location: ForecastLocation, hash: ForecastHash): ForecastKey =
    CoreForecastKey(location = location, hash = hash)

private data class CoreForecastKey(
    override val location: ForecastLocation,
    override val hash: ForecastHash,
) : ForecastKey


public sealed interface ForecastHash {
    public data object None : ForecastHash

    @JvmInline
    public value class Hash(public val value: String) : ForecastHash {
        override fun toString(): String = value
    }
}


public fun ForecastHash.toIfNoneMatchHeaderValue(): String = when (this) {
    is ForecastHash.Hash -> value
    ForecastHash.None -> "*"
}

private data class CoreForecastRequest(
    override val location: ForecastLocation,
) : ForecastRequest {}