/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.serialization

import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.LocalTime
import kotlinx.datetime.format.char
import kotlin.time.Duration
import kotlin.time.Duration.Companion.hours
import kotlin.time.Duration.Companion.minutes
import kotlin.time.Duration.Companion.seconds

public fun interface Deserializer<T> {
    public operator fun invoke(value: String): T
}

public object CoreDeserializer {
    public val localDateTime: Deserializer<LocalDateTime> = CoreLocalDateTimeDeserializer

    public val duration: Deserializer<Duration> = CoreDurationDeserializer
}


private object CoreLocalDateTimeDeserializer : Deserializer<LocalDateTime> {

    override fun invoke(value: String): LocalDateTime {
        return formatter.parse(value)
    }

    private val formatter = LocalDateTime.Format {
        date(LocalDate.Formats.ISO)
        char('T')
        hour()
        char(':')
        minute()
        char(':')
        second()
        chars("+00:00")
    }

}


private object CoreDurationDeserializer : Deserializer<Duration> {
    override fun invoke(value: String): Duration {
        val time = LocalTime.parse(value)
        return (time.hour.hours + time.minute.minutes + time.second.seconds)
    }
}