/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
@file:OptIn(ExperimentalTime::class)

package ventures.dvc.karbon.operation

import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import ventures.dvc.karbon.emission.ForecastEmission
import kotlin.time.Duration
import kotlin.time.ExperimentalTime
import kotlin.time.Instant


public fun ForecastEmission.toEmissionEntry(): EmissionEntry =
    EmissionEntry(
        time = time.toInstant(TimeZone.UTC),
        rating = rating,
        duration = duration
    )


public interface EmissionEntry {
    public val time: Instant
    public val rating: Double
    public val duration: Duration
}

public fun EmissionEntry(time: Instant, rating: Double, duration: Duration): EmissionEntry =
    EmissionEntryDto(
        time = time,
        rating = rating,
        duration = duration
    )

public val EmissionEntry.ratingPerMinute: Double
    get() {
        if (rating == 0.0) return 0.0
        if (duration.inWholeSeconds == 0L) return 0.0
        return rating / duration.inWholeMinutes
    }

public val EmissionEntry.endTime: Instant
    get() = time.plus(duration)


private data class EmissionEntryDto(
    override val time: Instant,
    override val rating: Double,
    override val duration: Duration,
) : EmissionEntry

