/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.provider.ktor

import arrow.core.nonFatalOrThrow
import io.ktor.client.*
import io.ktor.client.request.*
import io.ktor.client.statement.*
import ventures.dvc.karbon.location.ForecastLocation
import ventures.dvc.karbon.provider.*

internal class KtorValidateKeyOperation(
    private val configuration: KtorForecastProviderConfiguration,
    private val client: HttpClient,
) : ValidateKeyProvider {

    override suspend fun invoke(request: ValidateKeyRequest): ValidateKeyResponse {
        return try {
            validateKey(request)
        } catch (t: Throwable) {
            t.nonFatalOrThrow()
            ValidateKeyResponse.None(request.key.location)
        }
    }

    private suspend fun validateKey(request: ValidateKeyRequest): ValidateKeyResponse {
        val location = request.key.location
        val hash = request.key.hash

        val response = client.head(configuration.forecastsUrl(location)) {
            headers.ifNoneMatch(hash)
        }

        return when {
            response.isOk() -> response.toModifiedOrNone(location)
            response.isNotModified() -> ValidateKeyResponse.Valid(location, hash)
            else -> ValidateKeyResponse.None(location)
        }
    }
}

private fun HttpResponse.toModifiedOrNone(location: ForecastLocation) =
    when (val hash = forecastHash()) {
        is ForecastHash.Hash -> ValidateKeyResponse.Modified(location, hash)
        ForecastHash.None -> ValidateKeyResponse.None(location)
    }
