/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.provider.ktor

import arrow.core.Either
import arrow.core.left
import arrow.core.nonFatalOrThrow
import arrow.core.right
import io.ktor.client.*
import io.ktor.client.request.*
import io.ktor.client.statement.*
import ventures.dvc.karbon.location.ForecastLocation
import ventures.dvc.karbon.provider.*
import ventures.dvc.karbon.serialization.KarbonForecastDeserializer

internal class KtorForecastOperation(
    private val configuration: KtorForecastProviderConfiguration,
    private val serializer: KarbonForecastDeserializer,
    private val client: HttpClient,
) : KarbonForecastProvider {


    override suspend fun invoke(request: ForecastRequest): Either<ForecastError, ForecastResponse> {
        return try {
            fetchForecast(request)
        } catch (t: Throwable) {
            t.nonFatalOrThrow()
            ForecastError.Generic(t).left()
        }
    }

    private suspend fun fetchForecast(request: ForecastRequest): Either<ForecastError, ForecastResponse> {
        val location = request.location

        val response = client.get(configuration.forecastsUrl(location)) {}

        return when {
            response.isOk() -> response.deserializeToForecastResponse(location).right()
            response.isNotFound() -> ForecastError.LocationNotFound(location).left()
            else -> ForecastError.NotAvailable(location, response.status.description).left()
        }
    }

    private suspend fun HttpResponse.deserializeToForecastResponse(location: ForecastLocation): ForecastResponse {
        val content = serializer { bodyAsText() }
        return ForecastResponse(
            key = ForecastKey(location = location, hash = forecastHash()),
            emissions = content.emissions
        )
    }
}