/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.serialization

import kotlinx.datetime.LocalDateTime
import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.Json
import ventures.dvc.karbon.emission.ForecastEmission
import ventures.dvc.karbon.emission.ForecastEmissionContainer
import kotlin.time.Duration

public fun kotlinForecastDeserializer(): KarbonForecastDeserializer {
    return KotlinForecastDeserializer()
}

private class KotlinForecastDeserializer : KarbonForecastDeserializer {

    private val json = Json { ignoreUnknownKeys = true }

    override suspend fun invoke(source: suspend () -> String): ForecastEmissionContainer {
        return json.decodeFromString<KotlinForecastEmissionContainer>(source())
    }
}


@Serializable
private data class KotlinForecastEmissionContainer(
    @SerialName("Emissions")
    override val emissions: List<KotlinForecastEmission>,
) : ForecastEmissionContainer


@Serializable
private data class KotlinForecastEmission(
    @Serializable(with = KotlinLocalDateTimeSerializer::class)
    @SerialName("Time")
    override val time: LocalDateTime,
    @SerialName("Rating")
    override val rating: Double,
    @Serializable(with = KotlinDurationSerializer::class)
    @SerialName("Duration")
    override val duration: Duration,
) : ForecastEmission


private object KotlinDurationSerializer : KSerializer<Duration> {
    override val descriptor: SerialDescriptor =
        PrimitiveSerialDescriptor(
            "ventures.dvc.karbon.serialization.KotlinDurationSerializer",
            PrimitiveKind.STRING
        )

    override fun deserialize(decoder: Decoder): Duration {
        val value = decoder.decodeString()

        return CoreDeserializer.duration(value)
    }

    override fun serialize(encoder: Encoder, value: Duration) {
        encoder.encodeString(value.toString())
    }
}

internal object KotlinLocalDateTimeSerializer : KSerializer<LocalDateTime> {
    override val descriptor: SerialDescriptor = PrimitiveSerialDescriptor(
        "ventures.dvc.karbon.serialization.KotlinLocalDateTimeSerializer",
        PrimitiveKind.STRING
    )

    override fun deserialize(decoder: Decoder): LocalDateTime {
        val value = decoder.decodeString()
        return CoreDeserializer.localDateTime(value)
    }


    override fun serialize(encoder: Encoder, value: LocalDateTime) {
        encoder.encodeString(value.toString())
    }


}
