/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection;

import java.util.Comparator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.SortInsert;

/**
 * Сортировка списка
 * @author nt.gocha@gmail.com
 * @param <Container> Тип конейнера
 * @param <Element> Тип элемента
 */
public abstract class SortList<Container,Element>
    extends SortInsert<Container, Element>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(SortList.class.getName());
    
    private static Level logLevel(){ return logger.getLevel(); }
    private static boolean isLogSevere(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.SEVERE.intValue();
    }
    private static boolean isLogWarning(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.WARNING.intValue();
    }
    private static boolean isLogInfo(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.INFO.intValue();
    }
    private static boolean isLogFine(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.FINE.intValue();
    }
    private static boolean isLogFiner(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINER.intValue();
    }
    private static boolean isLogFinest(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINEST.intValue();
    }
    
    private static void logEntering(String method,Object ... args){
        logger.entering(SortList.class.getName(), method, args);
    }
    private static void logExiting(String method,Object result){
        logger.exiting(SortList.class.getName(), method, result);
    }
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }
    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }
    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }    
    //</editor-fold>
    
    protected long timeLimit = 0;

    /**
     * Ограничение по времени сортировки.
     * По умолчанию, без ограничения.
     * @return 0 и меньше - без ограничения
     */
    public long getTimeLimit() {
        return timeLimit;
    }

    /**
     * Ограничение по времени сортировки
     * @param timeLimit 0 и меньше - без ограничения
     */
    public void setTimeLimit(long timeLimit) {
        this.timeLimit = timeLimit;
    }
    
    /**
     * Сортировка контейнера
     * @param cont контейнер
     * @param comp компаратор
     * @param from начальный индекс
     * @param toEx конечный индекс искл
     */
    public void sort( Container cont, Comparator comp, int from, int toEx ){
        if (cont== null) {
            throw new IllegalArgumentException("cont==null");
        }
        if (comp== null) {
            throw new IllegalArgumentException("comp==null");
        }
        if( from > toEx ){
            throw new IllegalArgumentException("from("+from+") > toEx("+toEx+")");
        }
        
        int sortSize = toEx - from;
        
        // Сортировка не требуется
        if( sortSize<=1 )return;
        
        long tlimit = this.timeLimit;
        
        // рамка сортировки
        int sfrom = from;
        int sto = from+1;
        
        long t0 = System.currentTimeMillis();
        
        for( int elIdx=from+1; elIdx<toEx; elIdx++ ){
            if( Thread.interrupted() )break;
            if( tlimit>0 && ((System.currentTimeMillis() - t0)>=tlimit) )break;
            
            Element el = remove(cont, elIdx);
            if( el==null ){
                // вставка в начало списка
                insert(cont, from, el);
                
                // сдвиг рамки сортировки
                sfrom++;
                sto++;
                continue;
            }
            
            // вставка сортировкой
            sortInsert(cont, el, comp, sfrom, sto);
            sto++;
        }
    }
    
    /**
     * Удаление элемента из контейнера
     * @param cont Контейнер
     * @param index Индекс элемента в контейнере от нуля
     * @return Удаленный элемент
     */
    public abstract Element remove( Container cont, int index );
    
    /**
     * Создание экземпляра для работы со списком
     * @return экземпляр
     */
    public static SortList<List,Object> createForList(){
        return new SortList<List, Object>() {
            @Override
            public void insert(List container, int position, Object item) {
                container.add(position, item);
            }

            @Override
            public Object get(List container, int position) {
                return container.get(position);
            }

            @Override
            public Object remove(List cont, int index) {
                return cont.remove(index);
            }
        };
    }
}
