/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.graph;

/**
 * Базовый интервейс графа
 * @author GoCha
 */
public interface Graph<N,E>
{
    /**
     * Проверяет наличае вершины
     * @param node Вершина
     * @return true -вершина содержиться в графе
     */
    boolean contains(N node);
    
    /**
     * Добавляет вершину к графу
     * @param node Вершина
     */
    void add(N node);

    /**
     * Удаляет вершину из графа
     * @param node Вершина
     */
    void remove(N node);

    /**
     * Возвращает вершины графа
     * @return вершины
     */
    Iterable<N> getNodes();

    /**
     * Возвращает Ребра и вершины графа
     * @return Ребра и вершины
     */
    Iterable<Edge<N, E>> getEdges();
    
    /**
     * Возвращает ребра указанной вершины
     * @param node Вершина
     * @return ребра
     */
    Iterable<Edge<N, E>> edgesOf(N node);

    /**
     * Возвращает ребра из вершины А
     * @param nodeA Вершина А
     * @return Ребра
     */
    Iterable<Edge<N, E>> edgesOfNodeA(N nodeA);

    /**
     * Возвращает ребра из вершины Б
     * @param nodeB Вершина Б
     * @return Ребра
     */
    Iterable<Edge<N, E>> edgesOfNodeB(N nodeB);

    /**
     * Удаляет все ребра
     */
    void clearEdges();

    /**
     * Удалес все ребра и вершины
     */
    void clearAll();

    /**
     * Проверка наличия ребра между вершинами
     * @param a Вершина А
     * @param b Вершина Б
     * @return Флаг наличия ребра
     */
    boolean hasEdge(N a, N b);
    
    /**
     * Удаление ребра
     * @param a Вершина А
     * @param b Вершина Б
     */
    void removeEdge(N a, N b);

    /**
     * Возвращает ребро между вершинами
     * @param a Вершина А
     * @param b Вершина Б
     * @return Ребро
     */
    E getEdge(N a, N b);

    /**
     * Установка ребра между вершинами
     * @param a Вершина А
     * @param edge Ребро
     * @param b Вершина Б
     */
    public abstract void setEdge( N a, N b, E edge);

    /**
     * Возвращает ребра между вершинами
     * @param a Вершина А
     * @param b Вершина Б
     * @return Ребра
     */
    Iterable<E> getEdges(N a, N b);

    /**
     * Установка ребр между вершинами
     * @param a Вершина А
     * @param edges Ребра
     * @param b Вершина Б
     */
    public abstract void setEdges( N a, N b, Iterable<E> edges);
}
